/* This is xinterface.c
   A part of the Ymol program
   Copyright (C) 1997-1998 Daniel Spangberg
   */

static char rcsid[]="$Id: xinterface.c 140 2014-06-14 17:35:24Z daniels $";

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/X.h>

#ifdef USEOPENGL
#include <GL/gl.h>
#include <GL/glx.h>
#endif

#include "Xco.h"

#ifdef MULTIBUFFER
#include <X11/extensions/multibuf.h>
#endif
#ifdef DOUBLEBUFFER
#include <X11/extensions/Xdbe.h>
#endif

#include "xinterface.h"
#include "userinterface.h"
#include "drawevents.h"
#include "ccinterface.h"
#include "ymolbg.h"
#include "defaults.h"
#include "misc.h"
#include "phongdef.h"
#include "triangleworld.h"

static Pixmap ymolbg;
extern XcoObject topobject;
static int window_x,window_y;
static int banksize;

#ifdef MULTIBUFFER
static int use_multibuffer=0;
static int multibuffer_support=0;
static int multibuffer_initialized=0;
static Multibuffer multibuffers[2];
static int multibuffer_in_use=0;
#endif
#ifdef DOUBLEBUFFER
static int use_doublebuffer=0;
static int doublebuffer_support=0;
static int doublebuffer_initialized=0;
static XdbeBackBuffer backbuffer;
#endif


static XcoObject drawbox;
static unsigned char *image_buffer;
static uint32 *image_buffer32;
static uint32 *conv_image_buffer32;
static XImage *image;
static int32 *zbuffer_ptr;

static Pixmap pixmap_buffer[2];
static Pixmap stereo_pixmap[2];
static int active_pixmap=-1;

/*#define TILE */
#ifdef TILE
#include "cimages/tile.cmg"
#endif

#ifdef USEOPENGL

static int opengl_is_ok_to_use=0;
static GLXContext glxcontext;

int query_visual_gl(Display *dpy,int screen,XVisualInfo *vis)
{
    static int first_call=1;
    static int opengl_is_supported=0;
    if (first_call)
    {
	int errorBase,eventBase;
	if (glXQueryExtension(dpy,&errorBase,&eventBase))
	{
	    int major,minor;
	    
	    glXQueryVersion(dpy,&major,&minor);
	    printf("OpenGL (GLX) supported. GLX extension version: %d.%d\n",major,minor);
	    if ((major>=1) && (minor>=1))
	    {
		printf("%s\n",glXQueryServerString(dpy,screen,GLX_VENDOR));
	    }
	    opengl_is_supported=1;
	}
	first_call=0;
    }
    if (opengl_is_supported)
    {
	int value;
	/* printf("OpenGL queries visual %d: ",(int)myinfo[i].visualid); */
	glXGetConfig(dpy,vis,GLX_USE_GL,&value);
	/* printf("GLX_USE_GL=%d, ",value); */
	if (value)
	{
	    glXGetConfig(dpy,vis,GLX_RGBA,&value);
	    /* printf("GLX_RGBA=%d, ",value); */
	    if (value)
	    {
		glXGetConfig(dpy,vis,GLX_LEVEL,&value);
		/* printf("GLX_LEVEL=%d, ",value); */
		if (value==0)
		{
		    glXGetConfig(dpy,vis,GLX_DOUBLEBUFFER,&value);
		    /* printf("GLX_DOUBLEBUFFER=%d, ",value); */
		    if (value)
		    {
			glXGetConfig(dpy,vis,GLX_DEPTH_SIZE,&value);
				/* printf("GLX_DEPTH_SIZE=%d, ",value); */
			if (value>=16)
			{
			    opengl_is_ok_to_use=1;
			    return 1; /* Accepted... */
			}
			else
			    return 0;
		    }
		    else
			return 0;
		}
		else
		    return 0;
	    }
	    else
		return 0;
	}
	else
	    return 0;
    }
    else
	return 0;
}

void postprocess_visual_gl(Display *dpy,int screen,XVisualInfo *vis)
{
    if (opengl_is_ok_to_use)
    {
	glxcontext=glXCreateContext(dpy,vis,0,GL_TRUE);
	if (glXIsDirect(dpy,glxcontext))
	    printf("Direct rendering enabled.\n");
	else
	    printf("Direct rendering disabled (all commands through X-server.\n");
    }
}

int QueryOpenGLSupport()
{
  return(opengl_is_ok_to_use);
}

GLXContext GetGLXContext()
{
  return (glxcontext);
}
#endif

XcoObject get_drawbox()
{
  return drawbox;
}

static int the_draw_style=DRAW_STYLE_2D_CIRCLES;

void set_draw_style(int i)
{
  the_draw_style=i;
  if ((the_draw_style==DRAW_STYLE_2D_CIRCLES) ||
      (the_draw_style==DRAW_STYLE_WIREFRAME))
    {
      set_active_pixmap(0);
      set_oversampling(1);
    }
}

int get_draw_style()
{
  return(the_draw_style);
}

void set_bkgr_pixmap()
{
#if 0
  XSetWindowBackgroundPixmap(XcoGetDisplay(),XcoWindow(get_drawbox()),ymolbg);
#endif
  XClearWindow(XcoGetDisplay(),XcoWindow(get_drawbox()));
#if 0
  Arg tbarg[1];
  Cardinal tbcnt=0;
  XtSetArg(tbarg[tbcnt],XtNbackgroundPixmap,(XtArgVal) ymolbg);
  tbcnt++;
  XtSetValues(drawbox,tbarg,tbcnt);
#endif
}



void set_active_pixmap(int i)
{
  active_pixmap=i;
  if (active_pixmap==-1)
    {
      XcoSetBackground(get_drawbox(),XcoGetBackground(get_drawbox()));
      XClearWindow(XcoGetDisplay(),XcoWindow(get_drawbox()));
    }
  else
    {
#if 0
      XSetWindowBackgroundPixmap(XcoGetDisplay(),XcoWindow(get_drawbox()),pixmap_buffer[active_pixmap]);
      XClearWindow(XcoGetDisplay(),XcoWindow(get_drawbox()));
#endif
      XCopyArea(XcoGetDisplay(),
		pixmap_buffer[active_pixmap],XcoWindow(get_drawbox()),
		XcoGC(get_drawbox()),
		0,0,
		window_x,window_y,
		0,0);
    }
}

Pixmap get_active_pixmap()
{
  if (active_pixmap==-1)
    {
      printf("Request for active pixmap, when there is no active pixmap!\n");
      return (0);
    }
  else
    {
#ifdef DOUBLEBUFFER

#ifdef MULTIBUFFER
      /* MULTIBUFFER && DOUBLEBUFFER */
      if (use_doublebuffer)
	return(XcoWindow(drawbox));
      else if (use_multibuffer)
	return multibuffers[multibuffer_in_use];
      else
	return(pixmap_buffer[active_pixmap]);
#else
      /* DOUBLEBUFFER */
      if (use_doublebuffer)
	return(XcoWindow(drawbox));
      else
	return(pixmap_buffer[active_pixmap]);
#endif
#else
#ifdef MULTIBUFFER
      /* MULTIBUFFER */
      if (use_multibuffer)
	return multibuffers[multibuffer_in_use];
      else
	return(pixmap_buffer[active_pixmap]);
#else
      /* no extension */
      return(pixmap_buffer[active_pixmap]);
#endif
#endif
    }
}

static int current_stereo_pixmap;
static int current_is_stereo=0;

Pixmap get_current_draw_pixmap()
{
  if (current_is_stereo)
    return stereo_pixmap[current_stereo_pixmap];
  else
    return get_inactive_pixmap();
}

void set_stereo_mode(int i)
{
  current_is_stereo=i;
}

void set_stereo_buffer(int i)
{
  current_stereo_pixmap=i;
}

void copy_stereo_buffer()
{
  XCopyArea(XcoGetDisplay(),
	    stereo_pixmap[current_stereo_pixmap],
	    get_inactive_pixmap(),
	    XcoGC(get_drawbox()),
	    0,0,
	    window_x/2,window_y,
	    current_stereo_pixmap*window_x/2,0);
}

void clear_stereo_buffers(int ir, int ig, int ib)
{
  int i;
  XSetForeground(XcoGetDisplay(),XcoGC(get_drawbox()),XcoGetPixel32(0,0,PIXEL(ir,ig,ib)));
  for (i=0; i<=1; i++)
    XFillRectangle(XcoGetDisplay(),stereo_pixmap[i],XcoGC(get_drawbox()),
		   0,0,window_x/2,window_y);
}

Pixmap get_inactive_pixmap()
{
  if (active_pixmap==-1)
    {
      printf("Request for inactive pixmap, when there is no active pixmap!\n");
      return (0);
    }
  else
    {
#ifdef DOUBLEBUFFER

#ifdef MULTIBUFFER
      /* MULTIBUFFER && DOUBLEBUFFER */
      if (use_doublebuffer)
	return(backbuffer);
      else if (use_multibuffer)
	return multibuffers[1-multibuffer_in_use];
      else
	return(pixmap_buffer[1-active_pixmap]);
#else
      /* DOUBLEBUFFER */
      if (use_doublebuffer)
	return(backbuffer);
      else
	return(pixmap_buffer[1-active_pixmap]);
#endif
#else
#ifdef MULTIBUFFER
      /* MULTIBUFFER */
      if (use_multibuffer)
	return multibuffers[1-multibuffer_in_use];
      else
	return(pixmap_buffer[1-active_pixmap]);
#else
      /* no extension */
      return(pixmap_buffer[1-active_pixmap]);
#endif
#endif
    }
}

#ifdef MULTIBUFFER
void mbswap()
{
  multibuffer_in_use=1-multibuffer_in_use;
  XmbufDisplayBuffers(XcoGetDisplay(),1,multibuffers+multibuffer_in_use,0,0);
}
#endif

#ifdef DOUBLEBUFFER
void dbswap()
{
  XdbeSwapInfo i;
  i.swap_window=XcoWindow(drawbox);
  i.swap_action=XdbeUndefined;
  XdbeSwapBuffers(XcoGetDisplay(),&i,1);
}
#endif

void swap_pixmaps()
{
  if (the_draw_style==DRAW_STYLE_3D_SPHERES)
    {

#ifdef USEOPENGL
      if(get_use_opengl())
	{
	  glXSwapBuffers(XcoGetDisplay(),XcoWindow(drawbox));
	}
      else
	{
#endif


#ifdef DOUBLEBUFFER
#ifdef MULTIBUFFER
	  /* MULTIBUFFER && DOUBLEBUFFER */

	  if (use_doublebuffer)
	    {
	      dbswap();
	    }
	  else if (use_multibuffer)
	    {
	      mbswap();
	    }

#else
	  /* DOUBLEBUFFER */
	  if (use_doublebuffer)
	    {
	      dbswap();
	    }


#endif

#else
#ifdef MULTIBUFFER
	  /* MULTIBUFFER */
	  if (use_multibuffer)
	    {
	      mbswap();
	    }
      
#else
	  /* no extensions */

#endif
#endif
#ifdef USEOPENGL
	}
#endif


    }
  else /* 2D CIRCLES or WIREFRAME */
    {
      if (active_pixmap==-1)
	{
	  printf("Request to swap pixmaps, when there is no active pixmap!\n");
	}
      else
	{


#ifdef DOUBLEBUFFER
#ifdef MULTIBUFFER
	  /* MULTIBUFFER && DOUBLEBUFFER */

	  if (use_doublebuffer)
	    {
	      dbswap();
	    }
	  else if (use_multibuffer)
	    {
	      mbswap();
	    }
	  else
	    set_active_pixmap(1-active_pixmap);

#else
	  /* DOUBLEBUFFER */
	  if (use_doublebuffer)
	    {
	      dbswap();
	    }
	  else
	    set_active_pixmap(1-active_pixmap);


#endif

#else
#ifdef MULTIBUFFER
	  /* MULTIBUFFER */
	  if (use_multibuffer)
	    {
	      mbswap();
	    }
	  else
	    set_active_pixmap(1-active_pixmap);
      
#else
	  /* no extensions */
	  set_active_pixmap(1-active_pixmap);

#endif
#endif
	}
    }
}


void expose_draw_window()
{
  /*  printf("Window exposed...\n"); */
  if (the_draw_style==DRAW_STYLE_3D_SPHERES)
    {
#ifdef USEOPENGL
      if (get_use_opengl())
	{
	  fupd_();
	}
      else
	{
#endif
	  XPutImage(XcoGetDisplay(),XcoWindow(get_drawbox()),XcoGC(get_drawbox()),image,0,0,0,0,window_x,window_y);
	  XcoFlush();
#ifdef USEOPENGL
	}
#endif
	  

    }
  else
    {
      fupd_();
#if 0
#ifdef DOUBLEBUFFER
      if (use_doublebuffer)
	{
	  fupd_();
	}
#endif
#ifdef MULTIBUFFER
      if (use_multibuffer)
	{
	  fupd_();
	}
#endif
#endif
    }

}

static void compute_banksize(int oversample)
{
  banksize=(MAX_PIXELS)/(window_x*oversample*oversample);
  if (oversample!=1) 
    {
      banksize/=oversample;
      banksize*=oversample;
    }
  if ((oversample==1) && (banksize>window_y))
    banksize=window_y;

  if (banksize<oversample)
     banksize=oversample;
}

static void clear_pixmaps(int x,int y)
{
    int i;
    XSetForeground(XcoGetDisplay(),XcoGC(get_drawbox()),XcoGetPixel32(0,0,PIXEL(0,0,0)));
    for (i=0; i<=1; i++)
	XFillRectangle(XcoGetDisplay(),pixmap_buffer[i],XcoGC(get_drawbox()),
		       0,0,x,y);
}

void resize_draw_window(int x,int y)
{
    int oversample=0;
    int padbyte,pad,bpp;
    govsmp_(&oversample);
    /* printf("Window resized...%d,%d. Oversampling:%d\n",x,y,oversample); */
    /*Throw away old buffers: */
    free( conv_image_buffer32);
  
    XDestroyImage(image);
    free( image_buffer32);
    free( zbuffer_ptr);
    XFreePixmap(XcoGetDisplay(),pixmap_buffer[0]);
    XFreePixmap(XcoGetDisplay(),pixmap_buffer[1]);
  
    XFreePixmap(XcoGetDisplay(),stereo_pixmap[0]);
    XFreePixmap(XcoGetDisplay(),stereo_pixmap[1]);


    /* Create new buffers */
    window_x=x;
    window_y=y;
    compute_banksize(oversample);



    padbyte=XcoGetPadByte();
    pad=XcoGetPad();
    bpp=XcoGetBytesPerPixel();

    /* create image buffers */
    image_buffer=malloc((XcoGetNPad()+bpp*window_x)*window_y);
    image_buffer32=malloc(sizeof(uint32)*(window_x*banksize*oversample*oversample));
    zbuffer_ptr=malloc(sizeof(int32)*(window_x*banksize*oversample*oversample));
 
    conv_image_buffer32=malloc(sizeof( uint32)*(window_x*window_y));
  
    /* create pixmaps */
    pixmap_buffer[0]=XCreatePixmap(XcoGetDisplay(),XcoWindow(get_drawbox()),window_x,window_y,XcoGetDepth());
    pixmap_buffer[1]=XCreatePixmap(XcoGetDisplay(),XcoWindow(get_drawbox()),window_x,window_y,XcoGetDepth());
    stereo_pixmap[0]=XCreatePixmap(XcoGetDisplay(),XcoWindow(get_drawbox()),window_x/2,window_y,XcoGetDepth());
    stereo_pixmap[1]=XCreatePixmap(XcoGetDisplay(),XcoWindow(get_drawbox()),window_x/2,window_y,XcoGetDepth());


    image=XCreateImage(XcoGetDisplay(),XcoGetVisual(),XcoGetDepth(),ZPixmap,0,(char*) image_buffer,window_x,window_y,pad,window_x*bpp+XcoGetNPad());
  
    clear_pixmaps(window_x,window_y);

    reinif_(&x,&y);

}

void test_draw_window_size(int *x,int *y)
{
  int new_x=XcoGetObjectWidth(get_drawbox());
  int new_y=XcoGetObjectHeight(get_drawbox());
  int oversample=0;
  govsmp_(&oversample);
  

  if ((*x!=(new_x*oversample))||(*y!=(new_y*oversample)))
    {
      resize_draw_window(new_x,new_y);
    }
  
   *x=new_x;
   *y=new_y;

}

void set_oversampling(int oversample)
{
  sovsmp_(&oversample);
  resize_draw_window(window_x,window_y);
}



#ifdef MULTIBUFFER

#if 0
int query_multibuffer()
{
  int nbuf=XmbufCreateBuffers(XcoGetDisplay(),XcoWindow(drawbox),2,
			      MultibufferUpdateActionUndefined,
			      MultibufferUpdateHintFrequent,
			      multibuffers);
  printf("%d multibuffers found\n",nbuf);
  return(nbuf==2);
}
#endif
static int eventbase,errorbase;

int query_multibuffer()
{
  int status=XmbufQueryExtension(XcoGetDisplay(),&eventbase,&errorbase);
  if (status)
    {
      int maj,min;
      status=XmbufGetVersion(XcoGetDisplay(),&maj,&min);
      if (status)
	{
	  multibuffer_support=1;
	  printf("This server supports the multi buffer extension\nMulti buffer version %d.%d\n",maj,min);
	}
    }
  return(status);
}

int Iquery_mbuf_support()
{
  return (multibuffer_support);
}


void init_multibuffer()
{
  if (multibuffer_support)
    {
      if (!multibuffer_initialized)
	{
	  int cnt=XmbufCreateBuffers(XcoGetDisplay(),XcoWindow(drawbox),2,
				     MultibufferUpdateActionUndefined,
				     MultibufferUpdateHintFrequent,
				     multibuffers);
	  if (cnt!=2)
	    {
	      printf("Can't allocate the multibuffers.\n");
	      multibuffer_support=0;
	    }
	  else
	    {
	      multibuffer_initialized=1;
	    }
	}
      if (multibuffer_initialized)
	{
	  use_multibuffer=1;
#ifdef DOUBLEBUFFER
	  if (use_doublebuffer)
	    {
	      set_doublebuffer_usage(0);
	    }
#endif
	}
    }
  else
    {
      printf("This server doesn't support the multi buffer extension\n");
    }
}

void set_multibuffer_usage(int use)
{
  if (use)
    {
      init_multibuffer();
    }
  else
    {
      use_multibuffer=0;
    }
  if (use_multibuffer)
    printf("Using the multi buffer extension\n");
}

void switch_multibuffer_usage()
{
  set_multibuffer_usage(1-use_multibuffer);
}

#endif

#ifdef DOUBLEBUFFER
int query_doublebuffer()
{
  int maj,min;
  int status=XdbeQueryExtension(XcoGetDisplay(),&maj,&min);
  if (status)
    {
      doublebuffer_support=1;
      printf("This server supports the double buffer extension\nDouble buffer version %d.%d\n",maj,min);
    }
  return(status);
}

int Iquery_dbe_support()
{
  return (doublebuffer_support);
}

void init_doublebuffer()
{
  if (doublebuffer_support)
    {
      if (!doublebuffer_initialized)
	{
	  backbuffer=XdbeAllocateBackBufferName(XcoGetDisplay(),XcoWindow(drawbox),XdbeUndefined); 
	  doublebuffer_initialized=1;
	}
      use_doublebuffer=1;
#ifdef MULTIBUFFER
      if (use_multibuffer)
	{
	  set_multibuffer_usage(0);
	}
#endif
      
    }
  else
    {
      printf("This server doesn't support the double buffer extension\n");
    }
}

void set_doublebuffer_usage(int use)
{
  if (use)
    {
      init_doublebuffer();
    }
  else
    {
      use_doublebuffer=0;
    }
  if (use_doublebuffer)
    printf("Using the double buffer extension\n");
}

void switch_doublebuffer_usage()
{
  set_doublebuffer_usage(1-use_doublebuffer);
}


#endif



void drawbox_eventhandler(XcoObject id,XEvent event)
{
  switch (event.type)
    {
    case Expose:
      expose_draw_window();
      break;
    }
}

#define MENUHEIGHT 30

int init_animation_window(int *argc, char ***argv)
{
    XcoObject box3d;
    int bytes_per_pixel,padbyte,pad,oversample;
    int fwindow_x=720;  /* These currently get the animations in 720x576 size. Nice for PAL */
    int fwindow_y=576;
    if (*argc>3)
    {
	if (!strcmp((*argv)[1],"-s"))
	{
	    int i;
	    fwindow_x=strtol((*argv)[2],NULL,0);
	    fwindow_y=strtol((*argv)[3],NULL,0);
	    for (i=4; i<*argc; i++)
		(*argv)[i-3]=(*argv)[i];
	    (*argc)-=3;
	}
    }

    fwindow_x+=734-720; /* These currently get the animations in 720x576 size. Nice for PAL */
    fwindow_y+=615-576;
	
    XcoSetApplicationName("ymol");
    XcoSetApplicationClass("Ymol");

    topobject=XcoCreateNamedWindow(0,0,fwindow_x,fwindow_y,DEFAULT_BACKGROUND,1,-1,"Ymol");
    XSelectInput(XcoGetDisplay(),XcoWindow(topobject),
		 ExposureMask|ButtonPressMask|KeyPressMask|KeyReleaseMask|StructureNotifyMask);


#ifdef TILE
    Pixmap p=XcoCreatePixmapFromCimage(topobject,tile_cimage,tile_width,tile_height);
    XcoWindowTile(p);
    XcoSetBackgroundPixmap(topobject,p,True);
#endif



    box3d=XcoCreateBox3D(topobject,5,MENUHEIGHT,fwindow_x-10,fwindow_y-5-MENUHEIGHT,0,1);
    XSelectInput(XcoGetDisplay(),XcoWindow(box3d),
		 ExposureMask|ButtonPressMask);
    XcoSetResizeMethod(box3d,XcoUpLeftDownRight);

    /* initial window size */
    window_x=fwindow_x-10-4;
    window_y=fwindow_y-5-MENUHEIGHT-4;
#if 0 
    window_x=2048;
    window_y=2048;
#endif
    drawbox=XcoCreateHole(box3d,2,2,window_x,window_y);
#if 0
    XSelectInput(XcoGetDisplay(),XcoWindow(drawbox),
		 ExposureMask|ButtonPressMask|KeyPressMask|ButtonMotionMask|ButtonReleaseMask|OwnerGrabButtonMask);
#endif
    XSelectInput(XcoGetDisplay(),XcoWindow(drawbox),
		 ExposureMask|ButtonPressMask|ButtonMotionMask|ButtonReleaseMask|OwnerGrabButtonMask);

    XcoSetResizeMethod(drawbox,XcoUpLeftDownRight);



    bytes_per_pixel=XcoGetBytesPerPixel();
    padbyte=XcoGetPadByte();
    pad=XcoGetPad();

    /* Set oversampling: */
    oversample=2;
    sovsmp_(&oversample);

    compute_banksize(oversample);

    /* create image buffers */
    image_buffer=malloc((XcoGetNPad()+bytes_per_pixel*window_x)*window_y);
    image_buffer32=malloc(sizeof(uint32)*(window_x*banksize));
    zbuffer_ptr=malloc(sizeof(int32)*(window_x*banksize));
    conv_image_buffer32=malloc(sizeof( uint32)*(window_x*window_y));

    /* clear buffer... */
    /* memset(image_buffer,0,padbyte*window_x*window_y); */

    image=XCreateImage(XcoGetDisplay(),XcoGetVisual(),XcoGetDepth(),ZPixmap,0,(char*) image_buffer,
		       window_x,window_y,pad,window_x*bytes_per_pixel+XcoGetNPad());

    create_menus(topobject,fwindow_x); 



    ymolbg=get_ymolbg(drawbox);
    set_bkgr_pixmap();

    /* create pixmaps */
    pixmap_buffer[0]=XCreatePixmap(XcoGetDisplay(),XcoWindow(drawbox),
				   window_x,window_y,XcoGetDepth());
    pixmap_buffer[1]=XCreatePixmap(XcoGetDisplay(),XcoWindow(drawbox),
				   window_x,window_y,XcoGetDepth());
    stereo_pixmap[0]=XCreatePixmap(XcoGetDisplay(),XcoWindow(drawbox),
				   window_x/2,window_y,XcoGetDepth());
    stereo_pixmap[1]=XCreatePixmap(XcoGetDisplay(),XcoWindow(drawbox),
				   window_x/2,window_y,XcoGetDepth());

    clear_pixmaps(window_x,window_y);


    XcoAddCallback(topobject,event_draw_window);
    XcoAddCallback(drawbox,drawbox_eventhandler);
    XcoAddCallback(drawbox,event_draw_window);

#ifdef USEOPENGL
    if (QueryOpenGLSupport())
    {
	/* Bind window to opengl: */
	glXMakeCurrent(XcoGetDisplay(),XcoWindow(drawbox),GetGLXContext());

	glLightModeli(GL_LIGHT_MODEL_TWO_SIDE,0);
	glEnable(GL_LIGHTING);
	glEnable(GL_DEPTH_TEST);
	/*glShadeModel(GL_FLAT); */
	glShadeModel(GL_SMOOTH);
	glDepthFunc(GL_LESS);
    }
#endif

    return(0);
}

#if 0
void resize_window(int xsize, int ysize)
{
  /* Should resize the window. Create a new image buffer and a new image.*/
  window_x=xsize;
  window_y=ysize;
}
#endif
#if 0
void set_point(int x,int y, int r, int g, int b)
{
  XSetForeground(display,window_gc,get_pixel(x,y,r,g,b));
  XDrawPoint(display,window,window_gc,x,y);
}
#endif

uint32 *get_buffer_ptr()
{
  return(image_buffer32);
}

uint32 *get_conv_buffer_ptr()
{
  return(conv_image_buffer32);
}

int32 *get_zbuffer_ptr()
{
  return(zbuffer_ptr);
}

void get_buffer_size(int *x, int *y)
{
  int oversample=0;
  govsmp_(&oversample);
  *x=window_x*oversample;
  *y=window_y*oversample;
}

void clear_buffers(int icolor)
{
#ifdef USEOPENGL
  if (get_use_opengl())
    {
      int zclear;
      glClearColor(R_OF_PIXEL(icolor)*DIV255,
		   G_OF_PIXEL(icolor)*DIV255,
		   B_OF_PIXEL(icolor)*DIV255,
		   1.0);
      zclear=(window_x>window_y) ? window_x : window_y;
      glClearDepth(zclear);
      glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
      
    }
  else
    {
#endif
      int slen;
      int oversample=0;
      govsmp_(&oversample);
      slen=window_x*banksize*oversample*oversample;
      memsetint(zbuffer_ptr,-ZRES*ZMAXDEPTH,slen);
      memsetint((int*)image_buffer32,(int)icolor,slen);
#ifdef USEOPENGL
    }
#endif
}

void getbsz_(int *bsize)
{
  *bsize=banksize;
}

void doaa_(int *ystart,int *yend, int *istereo, int *nstereo)
{
  int oversample=0;
  govsmp_(&oversample);
  if (oversample==1)
    {
      if (*nstereo)
	{
	  int i;
	  int wxhalf=window_x/2;
	  for (i=*ystart; i<=*yend; i++)
	    {
	      memcpy(conv_image_buffer32+i*window_x+wxhalf*(*istereo),
		     image_buffer32+(i-*ystart)*window_x+wxhalf*(*istereo),
		     wxhalf*sizeof(int));
	    }
	}
      else
	memcpy(conv_image_buffer32+*ystart*window_x,image_buffer32,
	       window_x*(*yend-*ystart+1)*sizeof(int));
    }
  else
    {
      int wxhalf=window_x/(1+*nstereo);
      int wxstart=wxhalf*(*istereo);
      int w_s_y=*ystart/oversample;
      int w_e_y=*yend/oversample;
      uint32 *p=conv_image_buffer32+w_s_y*window_x;
      int o2=oversample*oversample;
      int wxo=window_x*oversample;
      int y,yp,x,xp,yo,xo,yoread;

      if (*nstereo)
	p+=wxhalf*(*istereo);

      for (y=w_s_y; y<=w_e_y; y++)
	{
	  yp=(y-w_s_y)*oversample;
	  for (x=wxstart; x<wxstart+wxhalf; x++)
	    {
	      int r=0,g=0,b=0;
	      xp=x*oversample;
	      for (yo=0; yo<oversample; yo++)
		{
		  yoread=(yp+yo)*wxo;
		  for (xo=0; xo<oversample; xo++)
		    {
		      r+=image_buffer32[yoread+xp+xo] & 0xFF;
		      g+=(image_buffer32[yoread+xp+xo]>>8) & 0xFF;
		      b+=(image_buffer32[yoread+xp+xo]>>16) & 0xFF;
		    }
		}
	      *p++=PIXEL(r/o2,g/o2,b/o2);
	    }
	  p+=wxhalf*(*nstereo);
	}
    }
}

void flush_image()
{
#ifdef USEOPENGL
  if (!get_use_opengl())
    {
#endif

#ifdef DOUBLEBUFFER

#ifdef MULTIBUFFER
      /* MULTIBUFFER && DOUBLEBUFFER */

      if (use_doublebuffer)
	{
	  XcoDitherImage(conv_image_buffer32, image_buffer,window_x,window_y,-1);
	  XPutImage(XcoGetDisplay(),backbuffer,XcoGC(drawbox),image,0,0,0,0,window_x,window_y);
	  XcoFlush();
	}
      else if (use_multibuffer)
	{
	  XcoDitherImage(conv_image_buffer32, image_buffer,window_x,window_y,-1);
	  XPutImage(XcoGetDisplay(),multibuffers[1-multibuffer_in_use],XcoGC(drawbox),image,0,0,0,0,window_x,window_y);
	  XcoFlush();
	}
      else
	{
	  XcoDitherImage(conv_image_buffer32, image_buffer,window_x,window_y,-1);
	  XPutImage(XcoGetDisplay(),XcoWindow(drawbox),XcoGC(drawbox),image,0,0,0,0,window_x,window_y);
	  XcoFlush();
	}
#else
      /* DOUBLEBUFFER */
      if (use_doublebuffer)
	{
	  XcoDitherImage(conv_image_buffer32, image_buffer,window_x,window_y,-1);
	  XPutImage(XcoGetDisplay(),backbuffer,XcoGC(drawbox),image,0,0,0,0,window_x,window_y);
	  XcoFlush();
	}
      else
	{
	  XcoDitherImage(conv_image_buffer32, image_buffer,window_x,window_y,-1);
	  XPutImage(XcoGetDisplay(),XcoWindow(drawbox),XcoGC(drawbox),image,0,0,0,0,window_x,window_y);
	  XcoFlush();
	}
#endif
#else
#ifdef MULTIBUFFER
      /* MULTIBUFFER */
      if (use_multibuffer)
	{
	  XcoDitherImage(conv_image_buffer32, image_buffer,window_x,window_y,-1);
	  XPutImage(XcoGetDisplay(),multibuffers[1-multibuffer_in_use],XcoGC(drawbox),image,0,0,0,0,window_x,window_y);
	  XcoFlush();
	}
      else
	{
	  XcoDitherImage(conv_image_buffer32, image_buffer,window_x,window_y,-1);
	  XPutImage(XcoGetDisplay(),XcoWindow(drawbox),XcoGC(drawbox),image,0,0,0,0,window_x,window_y);
	  XcoFlush();
	}
#else
      /* no extensions */
      XcoDitherImage(conv_image_buffer32, image_buffer,window_x,window_y,-1);
      XPutImage(XcoGetDisplay(),XcoWindow(drawbox),XcoGC(drawbox),image,0,0,0,0,window_x,window_y);
      XcoFlush();
#endif
#endif
#ifdef USEOPENGL
    }
#endif
}

void get_displayed_image()
{
  if (the_draw_style!=DRAW_STYLE_3D_SPHERES)
    {
      int x,y;
      XImage *myimage=XGetImage(XcoGetDisplay(),get_active_pixmap(),0,0,window_x,window_y,AllPlanes,ZPixmap);
      uint32 *ptr=conv_image_buffer32;
      for (y=0; y<window_y; y++)
	for (x=0; x<window_x; x++)
	    *ptr++=(uint32) XcoGetReversePixel((int)XGetPixel(myimage,x,y));
      XDestroyImage(myimage);
    }
}

