/* This is triangleworld.cc
   A part of the Ymol program
   Copyright (C) 1997-1998 Daniel Spangberg
   */

static char rcsid[]="$Id: triangleworld.c 88 2007-10-16 22:34:15Z daniels $";

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#ifdef USEOPENGL
#include <GL/gl.h>
#include <GL/glx.h>
#include "phongdef.h"
#endif

#include "xinterface.h"
#include "render_params.h"
#include "ccinterface.h"


#include "triangleworld.h"

/* if floating point sqrt is faster than fixed point define FLOAT_SQRT */

/* if HALF_P is defined some of the interpolated data will only be calculated
   one n:th of the time */
#define HALF_P

#define FIX_SHIFT2 4

/* TRIANGLE_PRECISION should be larger than the maximum size of a triangle
 TRIANGLE_SHIFT must be larger than FIX_SHIFT?
 FIX_SHIFT must be larger than 8 and less than or equal to 12 */

/* #define USE_LONG_LONG */

#ifdef USE_LONG_LONG
/* Quite untested. Probably slower if it works. */
#define FIX_SHIFT 24
#define FIX_PRECISION (1<<FIX_SHIFT)
#define TRIANGLE_SHIFT 25
#define TRIANGLE_PRECISION (1<<TRIANGLE_SHIFT)

typedef long long ttype;

#else

#define FIX_SHIFT 12
#define FIX_PRECISION (1<<FIX_SHIFT)
#define TRIANGLE_SHIFT 16
#define TRIANGLE_PRECISION (1<<TRIANGLE_SHIFT)

typedef int ttype;

#endif



#ifdef HALF_P

#define HALF_P_FACTOR 1
#define HALF_P_MASK 0x1

#endif



#ifndef PI
#define PI 3.1415926
#endif
#define TWOPI (2*PI)

#define tmin(a,b) ((a<b) ? (a) : (b))
#define tmin3(a,b,c) ((a<tmin(b,c)) ? (a) : (tmin(b,c)))
#define tmax(a,b) ((a>b) ? (a) : (b))
#define tmax3(a,b,c) ((a>tmax(b,c)) ? (a) : (tmax(b,c)))
#define fixabs(a) ((a>0) ? (a) : (-(a)))

static int ixtrans=0;

void stixtr_(int *i)
{
  ixtrans=*i;
}

#ifndef USEOPENGL
static int use_fast_sphere_emulation=1;
/* static int use_fast_sphere_emulation=0; */
static int mode_style_3d=MODE_STYLE_NATIVE;

#else

static int mode_style_3d=MODE_STYLE_OPENGL;
static int use_fast_sphere_emulation=0;
static int use_opengl=1;


void set_use_opengl(int i)
{
    if (!QueryOpenGLSupport())
    {
	mode_style_3d=MODE_STYLE_NATIVE;
	use_fast_sphere_emulation=1;
	use_opengl=0;
    }
    else
    {
	if (use_opengl!=i)
	{
	    use_opengl=i;
	    if (use_opengl)
		mode_style_3d=MODE_STYLE_OPENGL;
	    else
		mode_style_3d=MODE_STYLE_NATIVE;
	}
    }
}

int get_use_opengl()
{
    if (!QueryOpenGLSupport())
    {
	mode_style_3d=MODE_STYLE_NATIVE;
	use_fast_sphere_emulation=1;
	use_opengl=0;
    }
  return (use_opengl);
}

#endif

void set_3d_mode(int mode)
{
  mode_style_3d=mode;
}

int get_3d_mode()
{
  return(mode_style_3d);
}

void set_fast_emulation(int fast)
{
  use_fast_sphere_emulation=fast;
}

static int *fogtable=NULL;
static int ZMAXP,ZMINP;
static int fogtablesize;

static int use_perspective=1;

int get_perspective()
{
  return (use_perspective);
}

void set_perspective(int p)
{
  use_perspective=p;
}

static void compute_fog_table(int xsize,int ysize,double prop,double prop2)
{
  int i;
  if (fogtable!=NULL)
    free(fogtable);
  ZMAXP=tmax(xsize,ysize);
  ZMINP=-ZMAXP;
  fogtablesize=ZMAXP*2+1;
  fogtable=malloc(sizeof(int)*fogtablesize);
  for (i=ZMINP; i<=ZMAXP; i++)
    {
      double dv;
      fogfac_(&i,&xsize,&ysize,&prop,&prop2,&dv);
      fogtable[i-ZMINP]=dv*FIX_PRECISION;
    }
}


static triangle *triangle_hook=NULL;
static triangle **triangle_end=&triangle_hook;
static int ntriangles=0;
static triangle_point *tp_hook=NULL;
static int ntriangle_points=0;
static sphere *sphere_hook=NULL;
static int nspheres=0;
static cylinder *cylinder_hook=NULL;
static int ncylinders=0;

static lamp *lamps_nofp=NULL;
static int nlamps=0;
static int maxlamps=0;

static int bkgr_r,bkgr_g,bkgr_b;



void set_bglight(int r,int g,int b)
{
  bkgr_r=r;
  bkgr_g=g;
  bkgr_b=b;
}

void new_lamps(int number)
{
  if (maxlamps!=0)
    {
      free( lamps_nofp);
      nlamps=0;
    }
  lamps_nofp=malloc(sizeof( lamp)*(number));
  maxlamps=number;
}

void add_lamp(lamp l)
{
    if (nlamps>=maxlamps)
    {
	maxlamps++;
	lamps_nofp=realloc(lamps_nofp,maxlamps*sizeof *lamps_nofp);
    }
    
    if (nlamps<maxlamps)
    {
	lamps_nofp[nlamps]=l;
	nlamps++;
    }
}

int get_nlamps()
{
    return nlamps;
}

lamp *get_lamp(int i)
{
    return &lamps_nofp[i];
}

static void remove_tps()
{
  if (tp_hook!=NULL)
    {
      triangle_point *tp=tp_hook;
      triangle_point *next=tp->next_point;
      do {
	free( tp);
	if (next!=NULL)
	  {
	    tp=next;
	    next=next->next_point;
	  }
      } while (next!=NULL);
    }
  tp_hook=(triangle_point*)NULL;
  ntriangle_points=0;
}

static void remove_triangles()
{
  if (triangle_hook!=NULL)
    {
      triangle *t=triangle_hook;
      triangle *next=t->next_triangle;
      do {
	free( t);
	if (next!=NULL)
	  {
	    t=next;
	    next=next->next_triangle;
	  }
      } while (next!=NULL);
    }
  triangle_hook=(triangle*)NULL;
  ntriangles=0;
}

static void remove_spheres()
{
  if (sphere_hook!=NULL)
    {
      sphere *t=sphere_hook;
      sphere *next=t->next_sphere;
      do {
	free( t);
	if (next!=NULL)
	  {
	    t=next;
	    next=next->next_sphere;
	  }
      } while (next!=NULL);
    }
  sphere_hook=(sphere*)NULL;
  nspheres=0;
}

static void remove_cylinders()
{
  if (cylinder_hook!=NULL)
    {
      cylinder *t=cylinder_hook;
      cylinder *next=t->next_cylinder;
      do {
	free( t);
	if (next!=NULL)
	  {
	    t=next;
	    next=next->next_cylinder;
	  }
      } while (next!=NULL);
    }
  cylinder_hook=(cylinder*)NULL;
  ncylinders=0;
}

void new_triangles()
{
  remove_tps();
  remove_triangles();
  remove_spheres();
  remove_cylinders();
  triangle_end=&triangle_hook;
}

void register_triangle_point(triangle_point *tp)
{
  tp->next_point=tp_hook;
  tp_hook=tp;
  /*
  printf("Triangle point %d: x=%d, y=%d, z=%d\n",ntriangle_points,
	 tp->x,tp->y,tp->z);
	 */
  ntriangle_points++;
}

/* If all objects were sorted before adding the triangles so that
   objects in the back were added first, the triangles are almost
   perfectly sorted in the order front to back. This is bad, so 
   to handle transparent triangles the order must be changed,
   when inserting the triangles they should have the non apparent order.
   This is fixed by building the linked list in a non usual way (reverse order). */

void register_triangle(triangle *t)
{
#if 0
  t->next_triangle=triangle_hook;
  triangle_hook=t;
#endif
  t->next_triangle=NULL;
  *triangle_end=t;
  triangle_end=&t->next_triangle;
  ntriangles++;
}

void register_sphere(sphere *t)
{
  t->next_sphere=sphere_hook;
  sphere_hook=t;
  nspheres++;
}

void register_cylinder(cylinder *t)
{
  t->next_cylinder=cylinder_hook;
  cylinder_hook=t;
  ncylinders++;
}

typedef struct
{
  int x,y,z,r,g,b;
} lampfp;

static lampfp *lamps;




#if 0
/* Drawing a gouraud shaded, phong shaded triangle. */
static void draw_triangle(triangle *t,unsigned int *framebuffer,int *zbuffer,
			  int xsize,int ysize,
			  int ybankstart,int ybankend)
{
  int use_fogging=get_use_fogging();
  int fade_param=get_fading();
  int bkgrcol[3];
  get_bkgr_color_vector(&bkgrcol[0],&bkgrcol[1],&bkgrcol[2]);
  if ((t->point[0]->nz>0)||(t->point[1]->nz>0)||(t->point[2]->nz>0))
    {

#if 0
      double avtx=t->point[0]->nx+t->point[1]->nx+t->point[2]->nx;
      double avty=t->point[0]->ny+t->point[1]->ny+t->point[2]->ny;
      double avtz=t->point[0]->nz+t->point[1]->nz+t->point[2]->nz;
      double avl=sqrt(avtx*avtx+avty*avty+avtz*avtz);
      t->point[0]->nx=avtx/avl;
      t->point[1]->nx=avtx/avl;
      t->point[2]->nx=avtx/avl;
      t->point[0]->ny=avty/avl;
      t->point[1]->ny=avty/avl;
      t->point[2]->ny=avty/avl;
      t->point[0]->nz=avtz/avl;
      t->point[1]->nz=avtz/avl;
      t->point[2]->nz=avtz/avl;

#endif



      /* sort the triangles corners */
      /* kind of unrolled bubble sort */
      int ts[3]={0,1,2};
      if (t->point[0]->y>t->point[1]->y)
	{
	  ts[0]=1;
	  ts[1]=0;
	}
      if (t->point[ts[0]]->y>t->point[2]->y)
	{
	  ts[2]=ts[0];
	  ts[0]=2;
	}
      if (t->point[ts[1]]->y>t->point[ts[2]]->y)
	{
	  int iswp=ts[1];
	  ts[1]=ts[2];
	  ts[2]=iswp;
	}

      if ((t->point[ts[0]]->y<=ybankend) && (t->point[ts[2]]->y>=ybankstart))
	{

	  int xpmin=tmin3(t->point[0]->x,t->point[1]->x,t->point[2]->x);
	  int xpmax=tmax3(t->point[0]->x,t->point[1]->x,t->point[2]->x);
	  if ((xpmax>=0) && (xpmin<xsize))
	    {

	      /* the triangles sides are three lines. We will start with two of them and
		 most likely switch one side to the third one later on. */

	      ttype sline_x[3][2],sline_y[3][2];
	      ttype sline_z[3][2];
	      ttype sline_nx[3][2],sline_ny[3][2],sline_nz[3][2];

	      ttype sline_n[3][2];  
	      ttype sline_r[3][2],sline_g[3][2],sline_b[3][2];
	      ttype sline_fr[3][2];

	      ttype ymin,ymax,cline0y,cline0ymax,cline0x,cline0xdiff,yloop;

	      ttype cline0z,cline0zdiff;
	      ttype cline0n,cline0ndiff;
	      ttype cline0r,cline0rdiff;
	      ttype cline0g,cline0gdiff;
	      ttype cline0b,cline0bdiff;
	      ttype cline0nx,cline0nxdiff;
	      ttype cline0ny,cline0nydiff;
	      ttype cline0nz,cline0nzdiff;
	      ttype cline0fr,cline0frdiff;

	      int isswap;

	      ttype cline1y,cline1ymax,cline1x,cline1xdiff;

	      ttype cline1z,cline1zdiff;
	      ttype cline1n,cline1ndiff;
	      ttype cline1r,cline1rdiff;
	      ttype cline1g,cline1gdiff;
	      ttype cline1b,cline1bdiff;
	      ttype cline1nx,cline1nxdiff;
	      ttype cline1ny,cline1nydiff;
	      ttype cline1nz,cline1nzdiff;
	      ttype cline1fr,cline1frdiff;


	      sline_x[0][0]=t->point[ts[0]]->x;
	      sline_y[0][0]=t->point[ts[0]]->y;
	      sline_x[0][1]=t->point[ts[1]]->x;
	      sline_y[0][1]=t->point[ts[1]]->y;
	      sline_x[1][0]=t->point[ts[0]]->x;
	      sline_y[1][0]=t->point[ts[0]]->y;
	      sline_x[1][1]=t->point[ts[2]]->x;
	      sline_y[1][1]=t->point[ts[2]]->y;
	      sline_x[2][0]=t->point[ts[1]]->x;
	      sline_y[2][0]=t->point[ts[1]]->y;
	      sline_x[2][1]=t->point[ts[2]]->x;
	      sline_y[2][1]=t->point[ts[2]]->y;

	      /* Things that should be interpolated */

	      sline_z[0][0]=t->point[ts[0]]->z;
	      sline_z[0][1]=t->point[ts[1]]->z;
	      sline_z[1][0]=t->point[ts[0]]->z;
	      sline_z[1][1]=t->point[ts[2]]->z;
	      sline_z[2][0]=t->point[ts[1]]->z;
	      sline_z[2][1]=t->point[ts[2]]->z;
	      /* normals: */

	      sline_nx[0][0]=(int)(t->point[ts[0]]->nx*TRIANGLE_PRECISION);
	      sline_nx[0][1]=(int)(t->point[ts[1]]->nx*TRIANGLE_PRECISION);
	      sline_nx[1][0]=(int)(t->point[ts[0]]->nx*TRIANGLE_PRECISION);
	      sline_nx[1][1]=(int)(t->point[ts[2]]->nx*TRIANGLE_PRECISION);
	      sline_nx[2][0]=(int)(t->point[ts[1]]->nx*TRIANGLE_PRECISION);
	      sline_nx[2][1]=(int)(t->point[ts[2]]->nx*TRIANGLE_PRECISION);
	      sline_ny[0][0]=(int)(t->point[ts[0]]->ny*TRIANGLE_PRECISION);
	      sline_ny[0][1]=(int)(t->point[ts[1]]->ny*TRIANGLE_PRECISION);
	      sline_ny[1][0]=(int)(t->point[ts[0]]->ny*TRIANGLE_PRECISION);
	      sline_ny[1][1]=(int)(t->point[ts[2]]->ny*TRIANGLE_PRECISION);
	      sline_ny[2][0]=(int)(t->point[ts[1]]->ny*TRIANGLE_PRECISION);
	      sline_ny[2][1]=(int)(t->point[ts[2]]->ny*TRIANGLE_PRECISION);
	      sline_nz[0][0]=(int)(t->point[ts[0]]->nz*TRIANGLE_PRECISION);
	      sline_nz[0][1]=(int)(t->point[ts[1]]->nz*TRIANGLE_PRECISION);
	      sline_nz[1][0]=(int)(t->point[ts[0]]->nz*TRIANGLE_PRECISION);
	      sline_nz[1][1]=(int)(t->point[ts[2]]->nz*TRIANGLE_PRECISION);
	      sline_nz[2][0]=(int)(t->point[ts[1]]->nz*TRIANGLE_PRECISION);
	      sline_nz[2][1]=(int)(t->point[ts[2]]->nz*TRIANGLE_PRECISION);


	      /* exp */
	      sline_n[0][0]=t->point[ts[0]]->n;
	      sline_n[0][1]=t->point[ts[1]]->n;
	      sline_n[1][0]=t->point[ts[0]]->n;
	      sline_n[1][1]=t->point[ts[2]]->n;
	      sline_n[2][0]=t->point[ts[1]]->n;
	      sline_n[2][1]=t->point[ts[2]]->n;
	      /* rgbvalues: */

	      sline_r[0][0]=t->point[ts[0]]->r;
	      sline_r[0][1]=t->point[ts[1]]->r;
	      sline_r[1][0]=t->point[ts[0]]->r;
	      sline_r[1][1]=t->point[ts[2]]->r;
	      sline_r[2][0]=t->point[ts[1]]->r;
	      sline_r[2][1]=t->point[ts[2]]->r;
	      sline_g[0][0]=t->point[ts[0]]->g;
	      sline_g[0][1]=t->point[ts[1]]->g;
	      sline_g[1][0]=t->point[ts[0]]->g;
	      sline_g[1][1]=t->point[ts[2]]->g;
	      sline_g[2][0]=t->point[ts[1]]->g;
	      sline_g[2][1]=t->point[ts[2]]->g;
	      sline_b[0][0]=t->point[ts[0]]->b;
	      sline_b[0][1]=t->point[ts[1]]->b;
	      sline_b[1][0]=t->point[ts[0]]->b;
	      sline_b[1][1]=t->point[ts[2]]->b;
	      sline_b[2][0]=t->point[ts[1]]->b;
	      sline_b[2][1]=t->point[ts[2]]->b;
	      /* reflection coefficients: */

	      sline_fr[0][0]=(int)(t->point[ts[0]]->fr*TRIANGLE_PRECISION);
	      sline_fr[0][1]=(int)(t->point[ts[1]]->fr*TRIANGLE_PRECISION);
	      sline_fr[1][0]=(int)(t->point[ts[0]]->fr*TRIANGLE_PRECISION);
	      sline_fr[1][1]=(int)(t->point[ts[2]]->fr*TRIANGLE_PRECISION);
	      sline_fr[2][0]=(int)(t->point[ts[1]]->fr*TRIANGLE_PRECISION);
	      sline_fr[2][1]=(int)(t->point[ts[2]]->fr*TRIANGLE_PRECISION);




	      ymin=sline_y[0][0];
	      ymax=sline_y[2][1];
	      cline0y=sline_y[0][0];
	      cline0ymax=sline_y[0][1];
	      cline0x=sline_x[0][0];
	      cline0xdiff=0;


	      /* interpolate */
	      cline0z=sline_z[0][0];
	      cline0zdiff=0;
	      /* exp */
	      cline0n=sline_n[0][0];
	      cline0ndiff=0;
	      /* rgb */
	      cline0r=sline_r[0][0];
	      cline0rdiff=0;
	      cline0g=sline_g[0][0];
	      cline0gdiff=0;
	      cline0b=sline_b[0][0];
	      cline0bdiff=0;
	      /* normals */
	      cline0nx=sline_nx[0][0];
	      cline0nxdiff=0;
	      cline0ny=sline_ny[0][0];
	      cline0nydiff=0;
	      cline0nz=sline_nz[0][0];
	      cline0nzdiff=0;
	      /* refl. coeff. */
	      cline0fr=sline_fr[0][0];
	      cline0frdiff=0;

	      isswap=0;
	      if (ymin!=ymax)
		{
		  if (cline0ymax-cline0y!=0)
		    {
		      cline0xdiff=((sline_x[0][1]-cline0x)<<TRIANGLE_SHIFT)/
			(cline0ymax-cline0y);

		      /* interpolate */
		      cline0zdiff=((sline_z[0][1]-cline0z)<<TRIANGLE_SHIFT)/
			(cline0ymax-cline0y);
		      /* exp */
		      cline0ndiff=((sline_n[0][1]-cline0n)<<TRIANGLE_SHIFT)/
			(cline0ymax-cline0y);
		      /* rgb */
		      cline0rdiff=((sline_r[0][1]-cline0r)<<TRIANGLE_SHIFT)/
			(cline0ymax-cline0y);
		      cline0gdiff=((sline_g[0][1]-cline0g)<<TRIANGLE_SHIFT)/
			(cline0ymax-cline0y);
		      cline0bdiff=((sline_b[0][1]-cline0b)<<TRIANGLE_SHIFT)/
			(cline0ymax-cline0y);
		      /* normals */
		      cline0nxdiff=((sline_nx[0][1]-cline0nx))/
			(cline0ymax-cline0y);
		      cline0nydiff=((sline_ny[0][1]-cline0ny))/
			(cline0ymax-cline0y);
		      cline0nzdiff=((sline_nz[0][1]-cline0nz))/
			(cline0ymax-cline0y);
		      /* refl.coeff. */
		      cline0frdiff=((sline_fr[0][1]-cline0fr))/
			(cline0ymax-cline0y);
		    }
		  else
		    {
		      isswap=1;
		      cline0y=sline_y[2][0];
		      cline0ymax=sline_y[2][1];
		      cline0x=sline_x[2][0];

		      /* interpolate */
		      cline0z=sline_z[2][0];
		      /* exp */
		      cline0n=sline_n[2][0];
		      /* rgb */
		      cline0r=sline_r[2][0];
		      cline0g=sline_g[2][0];
		      cline0b=sline_b[2][0];
		      /* normals */
		      cline0nx=sline_nx[2][0];
		      cline0ny=sline_ny[2][0];
		      cline0nz=sline_nz[2][0];
		      /* refl.coeff. */
		      cline0fr=sline_fr[2][0];

		      if (cline0ymax-cline0y!=0)
			{
			  cline0xdiff=((sline_x[2][1]-cline0x)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);

			  /* interpolate */
			  cline0zdiff=((sline_z[2][1]-cline0z)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  /* exp */
			  cline0ndiff=((sline_n[2][1]-cline0n)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  /* rgb */
			  cline0rdiff=((sline_r[2][1]-cline0r)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  cline0gdiff=((sline_g[2][1]-cline0g)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  cline0bdiff=((sline_b[2][1]-cline0b)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  /* normals */
			  cline0nxdiff=((sline_nx[2][1]-cline0nx))/
			    (cline0ymax-cline0y);
			  cline0nydiff=((sline_ny[2][1]-cline0ny))/
			    (cline0ymax-cline0y);
			  cline0nzdiff=((sline_nz[2][1]-cline0nz))/
			    (cline0ymax-cline0y);
			  /* refl.coeff. */
			  cline0frdiff=((sline_fr[2][1]-cline0fr))/
			    (cline0ymax-cline0y);
			}
		      else
			{
			  /* drop this triangle... */
			  /* printf("Dropped triangle\n"); */
			  return;
			}
		    }
		}



	      cline1y=sline_y[1][0];
	      cline1ymax=sline_y[1][1];
	      cline1x=sline_x[1][0];
	      cline1xdiff=0;

	      /* interpolate */
	      cline1z=sline_z[1][0];
	      cline1zdiff=0;
	      /* exp */
	      cline1n=sline_n[1][0];
	      cline1ndiff=0;
	      /* rgb */
	      cline1r=sline_r[1][0];
	      cline1rdiff=0;
	      cline1g=sline_g[1][0];
	      cline1gdiff=0;
	      cline1b=sline_b[1][0];
	      cline1bdiff=0;
	      /* normals */
	      cline1nx=sline_nx[1][0];
	      cline1nxdiff=0;
	      cline1ny=sline_ny[1][0];
	      cline1nydiff=0;
	      cline1nz=sline_nz[1][0];
	      cline1nzdiff=0;
	      /* refl.coeff. */
	      cline1fr=sline_fr[1][0];
	      cline1frdiff=0;
	      if (ymin!=ymax)
		{

		  if (cline1ymax-cline1y!=0)
		    {
		      cline1xdiff=((sline_x[1][1]-cline1x)<<TRIANGLE_SHIFT)/
			(cline1ymax-cline1y);

		      /* interpolate */
		      cline1zdiff=((sline_z[1][1]-cline1z)<<TRIANGLE_SHIFT)/
			(cline1ymax-cline1y);
		      /* exp */
		      cline1ndiff=((sline_n[1][1]-cline1n)<<TRIANGLE_SHIFT)/
			(cline1ymax-cline1y);
		      /* rgb */
		      cline1rdiff=((sline_r[1][1]-cline1r)<<TRIANGLE_SHIFT)/
			(cline1ymax-cline1y);
		      cline1gdiff=((sline_g[1][1]-cline1g)<<TRIANGLE_SHIFT)/
			(cline1ymax-cline1y);
		      cline1bdiff=((sline_b[1][1]-cline1b)<<TRIANGLE_SHIFT)/
			(cline1ymax-cline1y);
		      /* normals */
		      cline1nxdiff=((sline_nx[1][1]-cline1nx))/
			(cline1ymax-cline1y);
		      cline1nydiff=((sline_ny[1][1]-cline1ny))/
			(cline1ymax-cline1y);
		      cline1nzdiff=((sline_nz[1][1]-cline1nz))/
			(cline1ymax-cline1y);
		      /* refl.coeff. */
		      cline1frdiff=((sline_fr[1][1]-cline1fr))/
			(cline1ymax-cline1y);
		    }
		  else
		    {
		      if (isswap)
			{
			  /* printf("Can't swap line because it was already swapped!\n"); */
			  return;
			}
		      /* printf("Swapped one line:1\n"); */
		      isswap=1;
		      cline1y=sline_y[2][0];
		      cline1ymax=sline_y[2][1];
		      cline1x=sline_x[2][0];

		      /* interpolate */
		      cline1z=sline_z[2][0];
		      /* exp */
		      cline1n=sline_n[2][0];
		      /* rgb */
		      cline1r=sline_r[2][0];
		      cline1g=sline_g[2][0];
		      cline1b=sline_b[2][0];
		      /* normals */
		      cline1nx=sline_nx[2][0];
		      cline1ny=sline_ny[2][0];
		      cline1nz=sline_nz[2][0];
		      /* refl.coeff. */
		      cline1fr=sline_fr[2][0];

		      if (cline1ymax-cline1y!=0)
			{
			  cline1xdiff=((sline_x[2][1]-cline1x)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);


			  /* interpolate */
			  cline1zdiff=((sline_z[2][1]-cline1z)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  /* exp */
			  cline1ndiff=((sline_n[2][1]-cline1n)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  /* rgb */
			  cline1rdiff=((sline_r[2][1]-cline1r)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  cline1gdiff=((sline_g[2][1]-cline1g)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  cline1bdiff=((sline_b[2][1]-cline1b)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  /* normals */
			  cline1nxdiff=((sline_nx[2][1]-cline1nx))/
			    (cline1ymax-cline1y);
			  cline1nydiff=((sline_ny[2][1]-cline1ny))/
			    (cline1ymax-cline1y);
			  cline1nzdiff=((sline_nz[2][1]-cline1nz))/
			    (cline1ymax-cline1y);
			  /* refl.coeff. */
			  cline1frdiff=((sline_fr[2][1]-cline1fr))/
			    (cline1ymax-cline1y);
			}
		      else
			{
			  /* drop this triangle... */
			  /* printf("Dropped triangle\n"); */
			  return;
			}
		    }
		}

	      cline0x<<=TRIANGLE_SHIFT;
	      cline1x<<=TRIANGLE_SHIFT;

	      /* interpolate */
	      cline0z<<=TRIANGLE_SHIFT;
	      cline1z<<=TRIANGLE_SHIFT;
	      /* exp */
	      cline0n<<=TRIANGLE_SHIFT;
	      cline1n<<=TRIANGLE_SHIFT;
	      /* rgb */
	      cline0r<<=TRIANGLE_SHIFT;
	      cline1r<<=TRIANGLE_SHIFT;
	      cline0g<<=TRIANGLE_SHIFT;
	      cline1g<<=TRIANGLE_SHIFT;
	      cline0b<<=TRIANGLE_SHIFT;
	      cline1b<<=TRIANGLE_SHIFT;
	      /* normals already got desired precision */
	      /* reflection coefficients already got desired precision */

#ifdef HALF_P

	      cline0frdiff<<=HALF_P_FACTOR;
	      cline1frdiff<<=HALF_P_FACTOR;
	      cline0ndiff<<=HALF_P_FACTOR;
	      cline1ndiff<<=HALF_P_FACTOR;
	      cline0rdiff<<=HALF_P_FACTOR;
	      cline1rdiff<<=HALF_P_FACTOR;
	      cline0gdiff<<=HALF_P_FACTOR;
	      cline1gdiff<<=HALF_P_FACTOR;
	      cline0bdiff<<=HALF_P_FACTOR;
	      cline1bdiff<<=HALF_P_FACTOR;

#endif

	      for (yloop=ymin; yloop<=ymax; yloop++)
		{
		  if (!isswap)
		    {
		      if (cline0y>cline0ymax)
			{
			  isswap=1;
			  /* swap... */
			  cline0y=sline_y[2][0];
			  cline0ymax=sline_y[2][1];
			  cline0x=sline_x[2][0];
			  cline0xdiff=((sline_x[2][1]-cline0x)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  cline0x<<=TRIANGLE_SHIFT;

			  /* interpolate */
			  cline0z=sline_z[2][0];
			  cline0zdiff=((sline_z[2][1]-cline0z)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  cline0z<<=TRIANGLE_SHIFT;
			  /* exp */
			  cline0n=sline_n[2][0];
			  cline0ndiff=((sline_n[2][1]-cline0n)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  cline0n<<=TRIANGLE_SHIFT;
			  /* rgb */
			  cline0r=sline_r[2][0];
			  cline0rdiff=((sline_r[2][1]-cline0r)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  cline0r<<=TRIANGLE_SHIFT;
			  cline0g=sline_g[2][0];
			  cline0gdiff=((sline_g[2][1]-cline0g)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  cline0g<<=TRIANGLE_SHIFT;
			  cline0b=sline_b[2][0];
			  cline0bdiff=((sline_b[2][1]-cline0b)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  cline0b<<=TRIANGLE_SHIFT;
			  /* normals */
			  cline0nx=sline_nx[2][0];
			  cline0nxdiff=((sline_nx[2][1]-cline0nx))/
			    (cline0ymax-cline0y);
			  cline0ny=sline_ny[2][0];
			  cline0nydiff=((sline_ny[2][1]-cline0ny))/
			    (cline0ymax-cline0y);
			  cline0nz=sline_nz[2][0];
			  cline0nzdiff=((sline_nz[2][1]-cline0nz))/
			    (cline0ymax-cline0y);
			  /* refl.coeff. */
			  cline0fr=sline_fr[2][0];
			  cline0frdiff=((sline_fr[2][1]-cline0fr))/
			    (cline0ymax-cline0y);

#ifdef HALF_P

			  cline0frdiff<<=HALF_P_FACTOR;
			  cline0ndiff<<=HALF_P_FACTOR;
			  cline0rdiff<<=HALF_P_FACTOR;
			  cline0gdiff<<=HALF_P_FACTOR;
			  cline0bdiff<<=HALF_P_FACTOR;

#endif


			}
		      else if (cline1y>cline1ymax)
			{
			  isswap=1;
			  /* swap... */
			  cline1y=sline_y[2][0];
			  cline1ymax=sline_y[2][1];
			  cline1x=sline_x[2][0]<<TRIANGLE_SHIFT;
			  cline1xdiff=((sline_x[2][1]-cline1x)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  cline1x<<=TRIANGLE_SHIFT;

	      
			  /* interpolate */
			  cline1z=sline_z[2][0]<<TRIANGLE_SHIFT;
			  cline1zdiff=((sline_z[2][1]-cline1z)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  cline1z<<=TRIANGLE_SHIFT;
			  /* exp */
			  cline1n=sline_n[2][0]<<TRIANGLE_SHIFT;
			  cline1ndiff=((sline_n[2][1]-cline1n)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  cline1n<<=TRIANGLE_SHIFT;
			  /* rgb */
			  cline1r=sline_r[2][0];
			  cline1rdiff=((sline_r[2][1]-cline1r)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  cline1r<<=TRIANGLE_SHIFT;
			  cline1g=sline_g[2][0];
			  cline1gdiff=((sline_g[2][1]-cline1g)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  cline1g<<=TRIANGLE_SHIFT;
			  cline1b=sline_b[2][0];
			  cline1bdiff=((sline_b[2][1]-cline1b)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  cline1b<<=TRIANGLE_SHIFT;
			  /* normals */
			  cline1nx=sline_nx[2][0];
			  cline1nxdiff=((sline_nx[2][1]-cline1nx))/
			    (cline1ymax-cline1y);
			  cline1ny=sline_ny[2][0];
			  cline1nydiff=((sline_ny[2][1]-cline1ny))/
			    (cline1ymax-cline1y);
			  cline1nz=sline_nz[2][0];
			  cline1nzdiff=((sline_nz[2][1]-cline1nz))/
			    (cline1ymax-cline1y);
			  /* refl.coeff. */
			  cline1fr=sline_fr[2][0];
			  cline1frdiff=((sline_fr[2][1]-cline1fr))/
			    (cline1ymax-cline1y);

#ifdef HALF_P

			  cline1frdiff<<=HALF_P_FACTOR;
			  cline1ndiff<<=HALF_P_FACTOR;
			  cline1rdiff<<=HALF_P_FACTOR;
			  cline1gdiff<<=HALF_P_FACTOR;
			  cline1bdiff<<=HALF_P_FACTOR;

#endif



			}
		    }      


		  if ((yloop>=ybankstart)&&(yloop<=ybankend))
		    {

		      ttype cx0=cline0x>>TRIANGLE_SHIFT;
		      ttype cx1=cline1x>>TRIANGLE_SHIFT;


		      /* interpolate */
		      ttype cz0=cline0z;
		      ttype cz1=cline1z;
		      /* exp */
		      ttype cn0=cline0n;
		      ttype cn1=cline1n;
		      /* rgb */
		      ttype cr0=cline0r;
		      ttype cr1=cline1r;
		      ttype cg0=cline0g;
		      ttype cg1=cline1g;
		      ttype cb0=cline0b;
		      ttype cb1=cline1b;
		      /* normals */
		      ttype cnx0=cline0nx;
		      ttype cnx1=cline1nx;
		      ttype cny0=cline0ny;
		      ttype cny1=cline1ny;
		      ttype cnz0=cline0nz;
		      ttype cnz1=cline1nz;
		      /* refl.coeff. */
		      ttype cfr0=cline0fr;
		      ttype cfr1=cline1fr;

		      /* interpolate */
		      ttype czdiff=0;
		      /* exp */
		      ttype cndiff=0;
		      /* rgb */
		      ttype crdiff=0;
		      ttype cgdiff=0;
		      ttype cbdiff=0;
		      /* normals */
		      ttype cnxdiff=0;
		      ttype cnydiff=0;
		      ttype cnzdiff=0;
		      /* refl.coeff. */
		      ttype cfrdiff=0;
		      
		      int mypoint;
		      ttype xloop;


		      if (cx0>cx1)
			{
			  ttype iswap=cx0;
			  cx0=cx1;
			  cx1=iswap;


			  /* interpolate */
			  iswap=cz0;
			  cz0=cz1;
			  cz1=iswap;
			  /* exp */
			  iswap=cn0;
			  cn0=cn1;
			  cn1=iswap;
			  /* rgb */
			  iswap=cr0;
			  cr0=cr1;
			  cr1=iswap;
			  iswap=cg0;
			  cg0=cg1;
			  cg1=iswap;
			  iswap=cb0;
			  cb0=cb1;
			  cb1=iswap;
			  /* normals */
			  iswap=cnx0;
			  cnx0=cnx1;
			  cnx1=iswap;
			  iswap=cny0;
			  cny0=cny1;
			  cny1=iswap;
			  iswap=cnz0;
			  cnz0=cnz1;
			  cnz1=iswap;
			  /* refl.coeff. */
			  iswap=cfr0;
			  cfr0=cfr1;
			  cfr1=iswap;
			}


		      if (cx1!=cx0)
			{
			  /* interpolate */
			  czdiff=(cz1-cz0)/(cx1-cx0);
			  /* exp */
			  cndiff=(cn1-cn0)/(cx1-cx0);
			  /* rgb */
			  crdiff=(cr1-cr0)/(cx1-cx0);
			  cgdiff=(cg1-cg0)/(cx1-cx0);
			  cbdiff=(cb1-cb0)/(cx1-cx0);
			  /* normals */
			  cnxdiff=(cnx1-cnx0)/(cx1-cx0);
			  cnydiff=(cny1-cny0)/(cx1-cx0);
			  cnzdiff=(cnz1-cnz0)/(cx1-cx0);
			  /* refl.coeff. */
			  cfrdiff=(cfr1-cfr0)/(cx1-cx0);
			}

#ifdef HALF_P

		      cndiff<<=HALF_P_FACTOR;
		      crdiff<<=HALF_P_FACTOR;
		      cgdiff<<=HALF_P_FACTOR;
		      cbdiff<<=HALF_P_FACTOR;
		      cfrdiff<<=HALF_P_FACTOR;

#endif


		      mypoint=xsize*(yloop-ybankstart)+cx0;
		      for (xloop=cx0; xloop<=cx1; xloop++)
			{
			  if ((xloop>=0)&&(xloop<xsize))
			    {
		      

			      ttype myz=cz0>>TRIANGLE_SHIFT;
			      if (zbuffer[mypoint]<myz)
				{
				  /* exp */
				  ttype myn=cn0>>TRIANGLE_SHIFT;
				  /* rgb */
				  ttype myr=cr0>>TRIANGLE_SHIFT;
				  ttype myg=cg0>>TRIANGLE_SHIFT;
				  ttype myb=cb0>>TRIANGLE_SHIFT;
				  /* compute color */

				  /* fixed point: (actually mix of two different fixed point numbers: TRIANGLE and FIX) */

				  ttype col_r=bkgr_r*cfr0;
				  ttype col_g=bkgr_g*cfr0;
				  ttype col_b=bkgr_b*cfr0;
				  int ilamp;

				  zbuffer[mypoint]=myz;

				  /* extract values from interpolated parameters */
				  for (ilamp=0; ilamp<nlamps; ilamp++)
				    {
				      ttype fvx=(cnx0>>(TRIANGLE_SHIFT-1-FIX_SHIFT))-
					(lamps[ilamp].x>>(TRIANGLE_SHIFT-FIX_SHIFT));
				      ttype fvy=(cny0>>(TRIANGLE_SHIFT-1-FIX_SHIFT))-
					(lamps[ilamp].y>>(TRIANGLE_SHIFT-FIX_SHIFT));
				      ttype fvz=(cnz0>>(TRIANGLE_SHIFT-1-FIX_SHIFT))-
					(lamps[ilamp].z>>(TRIANGLE_SHIFT-FIX_SHIFT));

				      ttype fvs=((fvx*fvx)>>FIX_SHIFT)+
					((fvy*fvy)>>FIX_SHIFT)+
					((fvz*fvz)>>FIX_SHIFT);

				      /* Square root... */
				      ttype fvl=fvs;
				      ttype skaldiff,skalrefl;
#ifdef FLOAT_SQRT
				      double vl;
#endif
				      if (fvl>0)
					{
#ifndef FLOAT_SQRT
					  ttype fvlold;
					  do {
					    fvlold=fvl;
					    fvl=(fvlold>>1)+(fvs<<(FIX_SHIFT-1))/fvlold;
					  } while (((fvl-fvlold)>10)||(fvl-fvlold)<-10);
#endif
#ifdef FLOAT_SQRT
					  vl=sqrt((double)(fvs)/FIX_PRECISION);
					  fvl=(int)(vl*FIX_PRECISION);
#endif
					  fvz=(fvz<<FIX_SHIFT)/fvl;

					}
				      else
					{
					  fvz=0;
					}
				      skaldiff=(((lamps[ilamp].x>>(TRIANGLE_SHIFT-FIX_SHIFT))*
						       (cnx0>>(TRIANGLE_SHIFT-FIX_SHIFT)))>>FIX_SHIFT)+
					(((lamps[ilamp].y>>(TRIANGLE_SHIFT-FIX_SHIFT))*
					  (cny0>>(TRIANGLE_SHIFT-FIX_SHIFT)))>>FIX_SHIFT)+
					(((lamps[ilamp].z>>(TRIANGLE_SHIFT-FIX_SHIFT))*
					  (cnz0>>(TRIANGLE_SHIFT-FIX_SHIFT)))>>FIX_SHIFT);

				      if (skaldiff<0)
					skaldiff=0;

				      skalrefl=0;
				      if (fvz>=0)
					{
					  /* skalrefl=pow(vz,myn); */
					  ttype cpow=fvz;
					  ttype myn2=myn;
					  skalrefl=1<<FIX_SHIFT;
					  while (myn2!=0)
					    {
					      if (myn2 & 0x1)
						skalrefl=(skalrefl*cpow)>>FIX_SHIFT;
					      myn2>>=1;
					      cpow=(cpow*cpow)>>FIX_SHIFT;
					    }
					  /* skalrefl=fvz; */
					}

				      col_r+=lamps[ilamp].r*
					((((myr<<(FIX_SHIFT-8))*skaldiff)>>
					  (FIX_SHIFT*2-TRIANGLE_SHIFT))
					 +(((cfr0>>(TRIANGLE_SHIFT-FIX_SHIFT))*skalrefl)>>
					   (FIX_SHIFT*2-TRIANGLE_SHIFT)));
				      col_g+=lamps[ilamp].g*
					((((myg<<(FIX_SHIFT-8))*skaldiff)>>
					  (FIX_SHIFT*2-TRIANGLE_SHIFT))
					 +(((cfr0>>(TRIANGLE_SHIFT-FIX_SHIFT))*skalrefl)>>
					   (FIX_SHIFT*2-TRIANGLE_SHIFT)));
				      col_b+=lamps[ilamp].b*
					((((myb<<(FIX_SHIFT-8))*skaldiff)>>
					  (FIX_SHIFT*2-TRIANGLE_SHIFT))
					 +(((cfr0>>(TRIANGLE_SHIFT-FIX_SHIFT))*skalrefl)>>
					   (FIX_SHIFT*2-TRIANGLE_SHIFT)));
				    }
				  col_r>>=TRIANGLE_SHIFT;
				  col_g>>=TRIANGLE_SHIFT;
				  col_b>>=TRIANGLE_SHIFT;
				  if (col_r>255)
				    col_r=255;
				  if (col_g>255)
				    col_g=255;
				  if (col_b>255)
				    col_b=255;
				  if (use_fogging)
				    {
				      /* Fog factor */
				      int ff=fogtable[myz-ZMINP];
				      col_r=((col_r*ff)+bkgrcol[0]*(FIX_PRECISION-ff))>>FIX_SHIFT;
				      col_g=((col_g*ff)+bkgrcol[1]*(FIX_PRECISION-ff))>>FIX_SHIFT;
				      col_b=((col_b*ff)+bkgrcol[2]*(FIX_PRECISION-ff))>>FIX_SHIFT;
				    }
				  if (fade_param!=255)
				    {
				      col_r=((col_r*fade_param)+bkgrcol[0]*(255-fade_param))>>8;
				      col_g=((col_g*fade_param)+bkgrcol[1]*(255-fade_param))>>8;
				      col_b=((col_b*fade_param)+bkgrcol[2]*(255-fade_param))>>8;
				    }
				  framebuffer[mypoint]=(col_r)|((col_g)<<8)|((col_b)<<16);
				}
			    }

			  /* interpolate */
			  cz0+=czdiff;
			  /* normals  */
			  cnx0+=cnxdiff;
			  cny0+=cnydiff;
			  cnz0+=cnzdiff;

#ifdef HALF_P
			  if (mypoint&HALF_P_MASK)
			    {
#endif
			      /* exp */
			      cn0+=cndiff;
			      /* rgb */
			      cr0+=crdiff;
			      cg0+=cgdiff;
			      cb0+=cbdiff;
			      /* refl.coeff. */
			      cfr0+=cfrdiff;
#ifdef HALF_P

			    }
#endif

			  mypoint++;
			}
		    }
		  




		  cline0x+=cline0xdiff;
		  cline1x+=cline1xdiff;

		  /* interpolate */
		  cline0z+=cline0zdiff;
		  cline1z+=cline1zdiff;

#ifdef HALF_P

		  if (yloop&HALF_P_MASK)
		    {

#endif

		      /* exp */
		      cline0n+=cline0ndiff;
		      cline1n+=cline1ndiff;
		      /* rgb */
		      cline0r+=cline0rdiff;
		      cline1r+=cline1rdiff;
		      cline0g+=cline0gdiff;
		      cline1g+=cline1gdiff;
		      cline0b+=cline0bdiff;
		      cline1b+=cline1bdiff;
		      /* refl.coeff. */
		      cline0fr+=cline0frdiff;
		      cline1fr+=cline1frdiff;

#ifdef HALF_P
	      
		    }

#endif

		  /* normals */
		  cline0nx+=cline0nxdiff;
		  cline1nx+=cline1nxdiff;
		  cline0ny+=cline0nydiff;
		  cline1ny+=cline1nydiff;
		  cline0nz+=cline0nzdiff;
		  cline1nz+=cline1nzdiff;

		  cline0y++;
		  cline1y++;
		}
	    }
	}
    }
}

#endif

/* Drawing a gouraud shaded, phong shaded triangle. */
static void draw_triangle(triangle *t,unsigned int *framebuffer,int *zbuffer,
			  int xsize,int ysize,
			  int ybankstart,int ybankend)
{
  int use_fogging=get_use_fogging();
  int fade_param=get_fading();
  int bkgrcol[3];
  int twosided=t->point[0]->twosided;
  get_bkgr_color_vector(&bkgrcol[0],&bkgrcol[1],&bkgrcol[2]);

#if 0
  if ((twosided)||(t->point[0]->nz>0)||(t->point[1]->nz>0)||(t->point[2]->nz>0))
#endif
    {
      /* sort the triangles corners */
      /* kind of unrolled bubble sort */
      int ts[3]={0,1,2};

      if (t->point[0]->y>t->point[1]->y)
	{
	  ts[0]=1;
	  ts[1]=0;
	}
      if (t->point[ts[0]]->y>t->point[2]->y)
	{
	  ts[2]=ts[0];
	  ts[0]=2;
	}
      if (t->point[ts[1]]->y>t->point[ts[2]]->y)
	{
	  int iswp=ts[1];
	  ts[1]=ts[2];
	  ts[2]=iswp;
	}

      if ((t->point[ts[0]]->y<=ybankend) && (t->point[ts[2]]->y>=ybankstart))
	{

	  int xpmin=tmin3(t->point[0]->x,t->point[1]->x,t->point[2]->x);
	  int xpmax=tmax3(t->point[0]->x,t->point[1]->x,t->point[2]->x);
	  if ((xpmax>=0) && (xpmin<xsize))
	    {

	      /* the triangles sides are three lines. We will start with two of them and
		 most likely switch one side to the third one later on. */

	      ttype sline_x[3][2],sline_y[3][2];
	      ttype sline_z[3][2];
	      ttype sline_nx[3][2],sline_ny[3][2],sline_nz[3][2];

	      ttype sline_n[3][2];  
	      ttype sline_r[3][2],sline_g[3][2],sline_b[3][2];
	      ttype sline_fr[3][2],sline_op[3][2];

	      ttype ymin,ymax,cline0y,cline0ymax,cline0x,cline0xdiff,yloop;

	      ttype cline0z,cline0zdiff;
	      ttype cline0n,cline0ndiff;
	      ttype cline0r,cline0rdiff;
	      ttype cline0g,cline0gdiff;
	      ttype cline0b,cline0bdiff;
	      ttype cline0nx,cline0nxdiff;
	      ttype cline0ny,cline0nydiff;
	      ttype cline0nz,cline0nzdiff;
	      ttype cline0fr,cline0frdiff;
	      ttype cline0op,cline0opdiff;

	      int isswap;

	      ttype cline1y,cline1ymax,cline1x,cline1xdiff;

	      ttype cline1z,cline1zdiff;
	      ttype cline1n,cline1ndiff;
	      ttype cline1r,cline1rdiff;
	      ttype cline1g,cline1gdiff;
	      ttype cline1b,cline1bdiff;
	      ttype cline1nx,cline1nxdiff;
	      ttype cline1ny,cline1nydiff;
	      ttype cline1nz,cline1nzdiff;
	      ttype cline1fr,cline1frdiff;
	      ttype cline1op,cline1opdiff;


	      sline_x[0][0]=t->point[ts[0]]->x;
	      sline_y[0][0]=t->point[ts[0]]->y;
	      sline_x[0][1]=t->point[ts[1]]->x;
	      sline_y[0][1]=t->point[ts[1]]->y;
	      sline_x[1][0]=t->point[ts[0]]->x;
	      sline_y[1][0]=t->point[ts[0]]->y;
	      sline_x[1][1]=t->point[ts[2]]->x;
	      sline_y[1][1]=t->point[ts[2]]->y;
	      sline_x[2][0]=t->point[ts[1]]->x;
	      sline_y[2][0]=t->point[ts[1]]->y;
	      sline_x[2][1]=t->point[ts[2]]->x;
	      sline_y[2][1]=t->point[ts[2]]->y;

	      /* Things that should be interpolated */

	      sline_z[0][0]=t->point[ts[0]]->z;
	      sline_z[0][1]=t->point[ts[1]]->z;
	      sline_z[1][0]=t->point[ts[0]]->z;
	      sline_z[1][1]=t->point[ts[2]]->z;
	      sline_z[2][0]=t->point[ts[1]]->z;
	      sline_z[2][1]=t->point[ts[2]]->z;
	      /* normals: */

	      sline_nx[0][0]=(int)(t->point[ts[0]]->nx*TRIANGLE_PRECISION);
	      sline_nx[0][1]=(int)(t->point[ts[1]]->nx*TRIANGLE_PRECISION);
	      sline_nx[1][0]=(int)(t->point[ts[0]]->nx*TRIANGLE_PRECISION);
	      sline_nx[1][1]=(int)(t->point[ts[2]]->nx*TRIANGLE_PRECISION);
	      sline_nx[2][0]=(int)(t->point[ts[1]]->nx*TRIANGLE_PRECISION);
	      sline_nx[2][1]=(int)(t->point[ts[2]]->nx*TRIANGLE_PRECISION);
	      sline_ny[0][0]=(int)(t->point[ts[0]]->ny*TRIANGLE_PRECISION);
	      sline_ny[0][1]=(int)(t->point[ts[1]]->ny*TRIANGLE_PRECISION);
	      sline_ny[1][0]=(int)(t->point[ts[0]]->ny*TRIANGLE_PRECISION);
	      sline_ny[1][1]=(int)(t->point[ts[2]]->ny*TRIANGLE_PRECISION);
	      sline_ny[2][0]=(int)(t->point[ts[1]]->ny*TRIANGLE_PRECISION);
	      sline_ny[2][1]=(int)(t->point[ts[2]]->ny*TRIANGLE_PRECISION);
	      sline_nz[0][0]=(int)(t->point[ts[0]]->nz*TRIANGLE_PRECISION);
	      sline_nz[0][1]=(int)(t->point[ts[1]]->nz*TRIANGLE_PRECISION);
	      sline_nz[1][0]=(int)(t->point[ts[0]]->nz*TRIANGLE_PRECISION);
	      sline_nz[1][1]=(int)(t->point[ts[2]]->nz*TRIANGLE_PRECISION);
	      sline_nz[2][0]=(int)(t->point[ts[1]]->nz*TRIANGLE_PRECISION);
	      sline_nz[2][1]=(int)(t->point[ts[2]]->nz*TRIANGLE_PRECISION);


	      /* exp */
	      sline_n[0][0]=t->point[ts[0]]->n;
	      sline_n[0][1]=t->point[ts[1]]->n;
	      sline_n[1][0]=t->point[ts[0]]->n;
	      sline_n[1][1]=t->point[ts[2]]->n;
	      sline_n[2][0]=t->point[ts[1]]->n;
	      sline_n[2][1]=t->point[ts[2]]->n;
	      /* rgbvalues: */

	      sline_r[0][0]=t->point[ts[0]]->r;
	      sline_r[0][1]=t->point[ts[1]]->r;
	      sline_r[1][0]=t->point[ts[0]]->r;
	      sline_r[1][1]=t->point[ts[2]]->r;
	      sline_r[2][0]=t->point[ts[1]]->r;
	      sline_r[2][1]=t->point[ts[2]]->r;
	      sline_g[0][0]=t->point[ts[0]]->g;
	      sline_g[0][1]=t->point[ts[1]]->g;
	      sline_g[1][0]=t->point[ts[0]]->g;
	      sline_g[1][1]=t->point[ts[2]]->g;
	      sline_g[2][0]=t->point[ts[1]]->g;
	      sline_g[2][1]=t->point[ts[2]]->g;
	      sline_b[0][0]=t->point[ts[0]]->b;
	      sline_b[0][1]=t->point[ts[1]]->b;
	      sline_b[1][0]=t->point[ts[0]]->b;
	      sline_b[1][1]=t->point[ts[2]]->b;
	      sline_b[2][0]=t->point[ts[1]]->b;
	      sline_b[2][1]=t->point[ts[2]]->b;
	      /* reflection coefficients: */

	      sline_fr[0][0]=(int)(t->point[ts[0]]->fr*TRIANGLE_PRECISION);
	      sline_fr[0][1]=(int)(t->point[ts[1]]->fr*TRIANGLE_PRECISION);
	      sline_fr[1][0]=(int)(t->point[ts[0]]->fr*TRIANGLE_PRECISION);
	      sline_fr[1][1]=(int)(t->point[ts[2]]->fr*TRIANGLE_PRECISION);
	      sline_fr[2][0]=(int)(t->point[ts[1]]->fr*TRIANGLE_PRECISION);
	      sline_fr[2][1]=(int)(t->point[ts[2]]->fr*TRIANGLE_PRECISION);

	      sline_op[0][0]=(int)(t->point[ts[0]]->opaq*TRIANGLE_PRECISION);
	      sline_op[0][1]=(int)(t->point[ts[1]]->opaq*TRIANGLE_PRECISION);
	      sline_op[1][0]=(int)(t->point[ts[0]]->opaq*TRIANGLE_PRECISION);
	      sline_op[1][1]=(int)(t->point[ts[2]]->opaq*TRIANGLE_PRECISION);
	      sline_op[2][0]=(int)(t->point[ts[1]]->opaq*TRIANGLE_PRECISION);
	      sline_op[2][1]=(int)(t->point[ts[2]]->opaq*TRIANGLE_PRECISION);




	      ymin=sline_y[0][0];
	      ymax=sline_y[2][1];
	      cline0y=sline_y[0][0];
	      cline0ymax=sline_y[0][1];
	      cline0x=sline_x[0][0];
	      cline0xdiff=0;


	      /* interpolate */
	      cline0z=sline_z[0][0];
	      cline0zdiff=0;
	      /* exp */
	      cline0n=sline_n[0][0];
	      cline0ndiff=0;
	      /* rgb */
	      cline0r=sline_r[0][0];
	      cline0rdiff=0;
	      cline0g=sline_g[0][0];
	      cline0gdiff=0;
	      cline0b=sline_b[0][0];
	      cline0bdiff=0;
	      /* normals */
	      cline0nx=sline_nx[0][0];
	      cline0nxdiff=0;
	      cline0ny=sline_ny[0][0];
	      cline0nydiff=0;
	      cline0nz=sline_nz[0][0];
	      cline0nzdiff=0;
	      /* refl. coeff. */
	      cline0fr=sline_fr[0][0];
	      cline0frdiff=0;
	      /* opacity */
	      cline0op=sline_op[0][0];
	      cline0opdiff=0;

	      isswap=0;
	      if (ymin!=ymax)
		{
		  if (cline0ymax-cline0y!=0)
		    {
		      cline0xdiff=((sline_x[0][1]-cline0x)<<TRIANGLE_SHIFT)/
			(cline0ymax-cline0y);

		      /* interpolate */
		      cline0zdiff=((sline_z[0][1]-cline0z)<<TRIANGLE_SHIFT)/
			(cline0ymax-cline0y);
		      /* exp */
		      cline0ndiff=((sline_n[0][1]-cline0n)<<TRIANGLE_SHIFT)/
			(cline0ymax-cline0y);
		      /* rgb */
		      cline0rdiff=((sline_r[0][1]-cline0r)<<TRIANGLE_SHIFT)/
			(cline0ymax-cline0y);
		      cline0gdiff=((sline_g[0][1]-cline0g)<<TRIANGLE_SHIFT)/
			(cline0ymax-cline0y);
		      cline0bdiff=((sline_b[0][1]-cline0b)<<TRIANGLE_SHIFT)/
			(cline0ymax-cline0y);
		      /* normals */
		      cline0nxdiff=((sline_nx[0][1]-cline0nx))/
			(cline0ymax-cline0y);
		      cline0nydiff=((sline_ny[0][1]-cline0ny))/
			(cline0ymax-cline0y);
		      cline0nzdiff=((sline_nz[0][1]-cline0nz))/
			(cline0ymax-cline0y);
		      /* refl.coeff. */
		      cline0frdiff=((sline_fr[0][1]-cline0fr))/
			(cline0ymax-cline0y);
		      /* opacity */
		      cline0opdiff=((sline_op[0][1]-cline0op))/
			(cline0ymax-cline0y);
		    }
		  else
		    {
		      isswap=1;
		      cline0y=sline_y[2][0];
		      cline0ymax=sline_y[2][1];
		      cline0x=sline_x[2][0];

		      /* interpolate */
		      cline0z=sline_z[2][0];
		      /* exp */
		      cline0n=sline_n[2][0];
		      /* rgb */
		      cline0r=sline_r[2][0];
		      cline0g=sline_g[2][0];
		      cline0b=sline_b[2][0];
		      /* normals */
		      cline0nx=sline_nx[2][0];
		      cline0ny=sline_ny[2][0];
		      cline0nz=sline_nz[2][0];
		      /* refl.coeff. */
		      cline0fr=sline_fr[2][0];
		      /* opacity */
		      cline0op=sline_op[2][0];

		      if (cline0ymax-cline0y!=0)
			{
			  cline0xdiff=((sline_x[2][1]-cline0x)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);

			  /* interpolate */
			  cline0zdiff=((sline_z[2][1]-cline0z)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  /* exp */
			  cline0ndiff=((sline_n[2][1]-cline0n)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  /* rgb */
			  cline0rdiff=((sline_r[2][1]-cline0r)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  cline0gdiff=((sline_g[2][1]-cline0g)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  cline0bdiff=((sline_b[2][1]-cline0b)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  /* normals */
			  cline0nxdiff=((sline_nx[2][1]-cline0nx))/
			    (cline0ymax-cline0y);
			  cline0nydiff=((sline_ny[2][1]-cline0ny))/
			    (cline0ymax-cline0y);
			  cline0nzdiff=((sline_nz[2][1]-cline0nz))/
			    (cline0ymax-cline0y);
			  /* refl.coeff. */
			  cline0frdiff=((sline_fr[2][1]-cline0fr))/
			    (cline0ymax-cline0y);
			  /* opacity */
			  cline0opdiff=((sline_op[2][1]-cline0op))/
			    (cline0ymax-cline0y);
			}
		      else
			{
			  /* drop this triangle... */
#if 0
			  printf("Dropped triangle\n");
#endif
			  return;
			}
		    }
		}



	      cline1y=sline_y[1][0];
	      cline1ymax=sline_y[1][1];
	      cline1x=sline_x[1][0];
	      cline1xdiff=0;

	      /* interpolate */
	      cline1z=sline_z[1][0];
	      cline1zdiff=0;
	      /* exp */
	      cline1n=sline_n[1][0];
	      cline1ndiff=0;
	      /* rgb */
	      cline1r=sline_r[1][0];
	      cline1rdiff=0;
	      cline1g=sline_g[1][0];
	      cline1gdiff=0;
	      cline1b=sline_b[1][0];
	      cline1bdiff=0;
	      /* normals */
	      cline1nx=sline_nx[1][0];
	      cline1nxdiff=0;
	      cline1ny=sline_ny[1][0];
	      cline1nydiff=0;
	      cline1nz=sline_nz[1][0];
	      cline1nzdiff=0;
	      /* refl.coeff. */
	      cline1fr=sline_fr[1][0];
	      cline1frdiff=0;
	      /* opacity */
	      cline1op=sline_op[1][0];
	      cline1opdiff=0;
	      if (ymin!=ymax)
		{

		  if (cline1ymax-cline1y!=0)
		    {
		      cline1xdiff=((sline_x[1][1]-cline1x)<<TRIANGLE_SHIFT)/
			(cline1ymax-cline1y);

		      /* interpolate */
		      cline1zdiff=((sline_z[1][1]-cline1z)<<TRIANGLE_SHIFT)/
			(cline1ymax-cline1y);
		      /* exp */
		      cline1ndiff=((sline_n[1][1]-cline1n)<<TRIANGLE_SHIFT)/
			(cline1ymax-cline1y);
		      /* rgb */
		      cline1rdiff=((sline_r[1][1]-cline1r)<<TRIANGLE_SHIFT)/
			(cline1ymax-cline1y);
		      cline1gdiff=((sline_g[1][1]-cline1g)<<TRIANGLE_SHIFT)/
			(cline1ymax-cline1y);
		      cline1bdiff=((sline_b[1][1]-cline1b)<<TRIANGLE_SHIFT)/
			(cline1ymax-cline1y);
		      /* normals */
		      cline1nxdiff=((sline_nx[1][1]-cline1nx))/
			(cline1ymax-cline1y);
		      cline1nydiff=((sline_ny[1][1]-cline1ny))/
			(cline1ymax-cline1y);
		      cline1nzdiff=((sline_nz[1][1]-cline1nz))/
			(cline1ymax-cline1y);
		      /* refl.coeff. */
		      cline1frdiff=((sline_fr[1][1]-cline1fr))/
			(cline1ymax-cline1y);
		      /* opacity */
		      cline1opdiff=((sline_op[1][1]-cline1op))/
			(cline1ymax-cline1y);
		    }
		  else
		    {
		      if (isswap)
			{
			  /* printf("Can't swap line because it was already swapped!\n"); */
			  return;
			}
		      /* printf("Swapped one line:1\n"); */
		      isswap=1;
		      cline1y=sline_y[2][0];
		      cline1ymax=sline_y[2][1];
		      cline1x=sline_x[2][0];

		      /* interpolate */
		      cline1z=sline_z[2][0];
		      /* exp */
		      cline1n=sline_n[2][0];
		      /* rgb */
		      cline1r=sline_r[2][0];
		      cline1g=sline_g[2][0];
		      cline1b=sline_b[2][0];
		      /* normals */
		      cline1nx=sline_nx[2][0];
		      cline1ny=sline_ny[2][0];
		      cline1nz=sline_nz[2][0];
		      /* refl.coeff. */
		      cline1fr=sline_fr[2][0];
		      /* opacity */
		      cline1op=sline_op[2][0];

		      if (cline1ymax-cline1y!=0)
			{
			  cline1xdiff=((sline_x[2][1]-cline1x)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);


			  /* interpolate */
			  cline1zdiff=((sline_z[2][1]-cline1z)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  /* exp */
			  cline1ndiff=((sline_n[2][1]-cline1n)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  /* rgb */
			  cline1rdiff=((sline_r[2][1]-cline1r)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  cline1gdiff=((sline_g[2][1]-cline1g)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  cline1bdiff=((sline_b[2][1]-cline1b)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  /* normals */
			  cline1nxdiff=((sline_nx[2][1]-cline1nx))/
			    (cline1ymax-cline1y);
			  cline1nydiff=((sline_ny[2][1]-cline1ny))/
			    (cline1ymax-cline1y);
			  cline1nzdiff=((sline_nz[2][1]-cline1nz))/
			    (cline1ymax-cline1y);
			  /* refl.coeff. */
			  cline1frdiff=((sline_fr[2][1]-cline1fr))/
			    (cline1ymax-cline1y);
			  /* opacity */
			  cline1opdiff=((sline_op[2][1]-cline1op))/
			    (cline1ymax-cline1y);
			}
		      else
			{
			  /* drop this triangle... */
#if 0
			  printf("Dropped triangle\n");
#endif
			  return;
			}
		    }
		}

	      cline0x<<=TRIANGLE_SHIFT;
	      cline1x<<=TRIANGLE_SHIFT;

	      /* interpolate */
	      cline0z<<=TRIANGLE_SHIFT;
	      cline1z<<=TRIANGLE_SHIFT;
	      /* exp */
	      cline0n<<=TRIANGLE_SHIFT;
	      cline1n<<=TRIANGLE_SHIFT;
	      /* rgb */
	      cline0r<<=TRIANGLE_SHIFT;
	      cline1r<<=TRIANGLE_SHIFT;
	      cline0g<<=TRIANGLE_SHIFT;
	      cline1g<<=TRIANGLE_SHIFT;
	      cline0b<<=TRIANGLE_SHIFT;
	      cline1b<<=TRIANGLE_SHIFT;
	      /* normals already got desired precision */
	      /* reflection coefficients already got desired precision */
	      /* opacity ... */

#ifdef HALF_P

	      cline0frdiff<<=HALF_P_FACTOR;
	      cline1frdiff<<=HALF_P_FACTOR;
	      cline0opdiff<<=HALF_P_FACTOR;
	      cline1opdiff<<=HALF_P_FACTOR;
	      cline0ndiff<<=HALF_P_FACTOR;
	      cline1ndiff<<=HALF_P_FACTOR;
	      cline0rdiff<<=HALF_P_FACTOR;
	      cline1rdiff<<=HALF_P_FACTOR;
	      cline0gdiff<<=HALF_P_FACTOR;
	      cline1gdiff<<=HALF_P_FACTOR;
	      cline0bdiff<<=HALF_P_FACTOR;
	      cline1bdiff<<=HALF_P_FACTOR;

#endif

	      for (yloop=ymin; yloop<=ymax; yloop++)
		{
		  if (!isswap)
		    {
		      if (cline0y>cline0ymax)
			{
			  isswap=1;
			  /* swap... */
			  cline0y=sline_y[2][0];
			  cline0ymax=sline_y[2][1];
			  cline0x=sline_x[2][0];
			  cline0xdiff=((sline_x[2][1]-cline0x)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  cline0x<<=TRIANGLE_SHIFT;

			  /* interpolate */
			  cline0z=sline_z[2][0];
			  cline0zdiff=((sline_z[2][1]-cline0z)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  cline0z<<=TRIANGLE_SHIFT;
			  /* exp */
			  cline0n=sline_n[2][0];
			  cline0ndiff=((sline_n[2][1]-cline0n)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  cline0n<<=TRIANGLE_SHIFT;
			  /* rgb */
			  cline0r=sline_r[2][0];
			  cline0rdiff=((sline_r[2][1]-cline0r)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  cline0r<<=TRIANGLE_SHIFT;
			  cline0g=sline_g[2][0];
			  cline0gdiff=((sline_g[2][1]-cline0g)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  cline0g<<=TRIANGLE_SHIFT;
			  cline0b=sline_b[2][0];
			  cline0bdiff=((sline_b[2][1]-cline0b)<<TRIANGLE_SHIFT)/
			    (cline0ymax-cline0y);
			  cline0b<<=TRIANGLE_SHIFT;
			  /* normals */
			  cline0nx=sline_nx[2][0];
			  cline0nxdiff=((sline_nx[2][1]-cline0nx))/
			    (cline0ymax-cline0y);
			  cline0ny=sline_ny[2][0];
			  cline0nydiff=((sline_ny[2][1]-cline0ny))/
			    (cline0ymax-cline0y);
			  cline0nz=sline_nz[2][0];
			  cline0nzdiff=((sline_nz[2][1]-cline0nz))/
			    (cline0ymax-cline0y);
			  /* refl.coeff. */
			  cline0fr=sline_fr[2][0];
			  cline0frdiff=((sline_fr[2][1]-cline0fr))/
			    (cline0ymax-cline0y);
			  /* opacity */
			  cline0op=sline_op[2][0];
			  cline0opdiff=((sline_op[2][1]-cline0op))/
			    (cline0ymax-cline0y);

#ifdef HALF_P

			  cline0frdiff<<=HALF_P_FACTOR;
			  cline0opdiff<<=HALF_P_FACTOR;
			  cline0ndiff<<=HALF_P_FACTOR;
			  cline0rdiff<<=HALF_P_FACTOR;
			  cline0gdiff<<=HALF_P_FACTOR;
			  cline0bdiff<<=HALF_P_FACTOR;

#endif


			}
		      else if (cline1y>cline1ymax)
			{
			  isswap=1;
			  /* swap... */
			  cline1y=sline_y[2][0];
			  cline1ymax=sline_y[2][1];
			  cline1x=sline_x[2][0]<<TRIANGLE_SHIFT;
			  cline1xdiff=((sline_x[2][1]-cline1x)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  cline1x<<=TRIANGLE_SHIFT;

	      
			  /* interpolate */
			  cline1z=sline_z[2][0]<<TRIANGLE_SHIFT;
			  cline1zdiff=((sline_z[2][1]-cline1z)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  cline1z<<=TRIANGLE_SHIFT;
			  /* exp */
			  cline1n=sline_n[2][0]<<TRIANGLE_SHIFT;
			  cline1ndiff=((sline_n[2][1]-cline1n)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  cline1n<<=TRIANGLE_SHIFT;
			  /* rgb */
			  cline1r=sline_r[2][0];
			  cline1rdiff=((sline_r[2][1]-cline1r)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  cline1r<<=TRIANGLE_SHIFT;
			  cline1g=sline_g[2][0];
			  cline1gdiff=((sline_g[2][1]-cline1g)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  cline1g<<=TRIANGLE_SHIFT;
			  cline1b=sline_b[2][0];
			  cline1bdiff=((sline_b[2][1]-cline1b)<<TRIANGLE_SHIFT)/
			    (cline1ymax-cline1y);
			  cline1b<<=TRIANGLE_SHIFT;
			  /* normals */
			  cline1nx=sline_nx[2][0];
			  cline1nxdiff=((sline_nx[2][1]-cline1nx))/
			    (cline1ymax-cline1y);
			  cline1ny=sline_ny[2][0];
			  cline1nydiff=((sline_ny[2][1]-cline1ny))/
			    (cline1ymax-cline1y);
			  cline1nz=sline_nz[2][0];
			  cline1nzdiff=((sline_nz[2][1]-cline1nz))/
			    (cline1ymax-cline1y);
			  /* refl.coeff. */
			  cline1fr=sline_fr[2][0];
			  cline1frdiff=((sline_fr[2][1]-cline1fr))/
			    (cline1ymax-cline1y);
			  /* opacity */
			  cline1op=sline_op[2][0];
			  cline1opdiff=((sline_op[2][1]-cline1op))/
			    (cline1ymax-cline1y);

#ifdef HALF_P

			  cline1frdiff<<=HALF_P_FACTOR;
			  cline1opdiff<<=HALF_P_FACTOR;
			  cline1ndiff<<=HALF_P_FACTOR;
			  cline1rdiff<<=HALF_P_FACTOR;
			  cline1gdiff<<=HALF_P_FACTOR;
			  cline1bdiff<<=HALF_P_FACTOR;

#endif



			}
		    }      


		  if ((yloop>=ybankstart)&&(yloop<=ybankend))
		    {

		      ttype cx0=cline0x>>TRIANGLE_SHIFT;
		      ttype cx1=cline1x>>TRIANGLE_SHIFT;


		      /* interpolate */
		      ttype cz0=cline0z;
		      ttype cz1=cline1z;
		      /* exp */
		      ttype cn0=cline0n;
		      ttype cn1=cline1n;
		      /* rgb */
		      ttype cr0=cline0r;
		      ttype cr1=cline1r;
		      ttype cg0=cline0g;
		      ttype cg1=cline1g;
		      ttype cb0=cline0b;
		      ttype cb1=cline1b;
		      /* normals */
		      ttype cnx0=cline0nx;
		      ttype cnx1=cline1nx;
		      ttype cny0=cline0ny;
		      ttype cny1=cline1ny;
		      ttype cnz0=cline0nz;
		      ttype cnz1=cline1nz;
		      /* refl.coeff. */
		      ttype cfr0=cline0fr;
		      ttype cfr1=cline1fr;
		      /* opacity */
		      ttype cop0=cline0op;
		      ttype cop1=cline1op;

		      /* interpolate */
		      ttype czdiff=0;
		      /* exp */
		      ttype cndiff=0;
		      /* rgb */
		      ttype crdiff=0;
		      ttype cgdiff=0;
		      ttype cbdiff=0;
		      /* normals */
		      ttype cnxdiff=0;
		      ttype cnydiff=0;
		      ttype cnzdiff=0;
		      /* refl.coeff. */
		      ttype cfrdiff=0;
		      /* opacity */
		      ttype copdiff=0;
		      
		      int mypoint;
		      ttype xloop;


		      if (cx0>cx1)
			{
			  ttype iswap=cx0;
			  cx0=cx1;
			  cx1=iswap;


			  /* interpolate */
			  iswap=cz0;
			  cz0=cz1;
			  cz1=iswap;
			  /* exp */
			  iswap=cn0;
			  cn0=cn1;
			  cn1=iswap;
			  /* rgb */
			  iswap=cr0;
			  cr0=cr1;
			  cr1=iswap;
			  iswap=cg0;
			  cg0=cg1;
			  cg1=iswap;
			  iswap=cb0;
			  cb0=cb1;
			  cb1=iswap;
			  /* normals */
			  iswap=cnx0;
			  cnx0=cnx1;
			  cnx1=iswap;
			  iswap=cny0;
			  cny0=cny1;
			  cny1=iswap;
			  iswap=cnz0;
			  cnz0=cnz1;
			  cnz1=iswap;
			  /* refl.coeff. */
			  iswap=cfr0;
			  cfr0=cfr1;
			  cfr1=iswap;
			  /* opacity */
			  iswap=cop0;
			  cop0=cop1;
			  cop1=iswap;
			}


		      if (cx1!=cx0)
			{
			  /* interpolate */
			  czdiff=(cz1-cz0)/(cx1-cx0);
			  /* exp */
			  cndiff=(cn1-cn0)/(cx1-cx0);
			  /* rgb */
			  crdiff=(cr1-cr0)/(cx1-cx0);
			  cgdiff=(cg1-cg0)/(cx1-cx0);
			  cbdiff=(cb1-cb0)/(cx1-cx0);
			  /* normals */
			  cnxdiff=(cnx1-cnx0)/(cx1-cx0);
			  cnydiff=(cny1-cny0)/(cx1-cx0);
			  cnzdiff=(cnz1-cnz0)/(cx1-cx0);
			  /* refl.coeff. */
			  cfrdiff=(cfr1-cfr0)/(cx1-cx0);
			  /* opacity */
			  copdiff=(cop1-cop0)/(cx1-cx0);
			}

#ifdef HALF_P

		      cndiff<<=HALF_P_FACTOR;
		      crdiff<<=HALF_P_FACTOR;
		      cgdiff<<=HALF_P_FACTOR;
		      cbdiff<<=HALF_P_FACTOR;
		      cfrdiff<<=HALF_P_FACTOR;
		      copdiff<<=HALF_P_FACTOR;

#endif


		      mypoint=xsize*(yloop-ybankstart)+cx0+ixtrans;
		      for (xloop=cx0+ixtrans; xloop<=cx1+ixtrans; xloop++)
			{
			  if ((xloop>=0)&&(xloop<xsize))
			    {
#if 0
			      if (twosided||(cnz0>0))
#endif
				{
				  ttype myz=cz0>>TRIANGLE_SHIFT;
				  int usereverse=0;
				  if (twosided && (cnz0<=0))
				    usereverse=1;
				  if (zbuffer[mypoint]<myz)
				    {
				      unsigned int oldfbuf;

				      /* exp */
				      ttype myn=cn0>>TRIANGLE_SHIFT;
				      /* rgb */
				      ttype myr=cr0>>TRIANGLE_SHIFT;
				      ttype myg=cg0>>TRIANGLE_SHIFT;
				      ttype myb=cb0>>TRIANGLE_SHIFT;
				      /* compute color */

				      /* fixed point: (actually mix of two different fixed point numbers: TRIANGLE and FIX) */

				      ttype col_r=bkgr_r*cfr0;
				      ttype col_g=bkgr_g*cfr0;
				      ttype col_b=bkgr_b*cfr0;
				      
				      ttype col_r_spec=0;
				      ttype col_g_spec=0;
				      ttype col_b_spec=0;

				      int ilamp;

				      zbuffer[mypoint]=myz;

				      /* extract values from interpolated parameters */
				      for (ilamp=0; ilamp<nlamps; ilamp++)
					{
					  ttype fvx,fvy,fvz;
					  ttype skaldiff,skalrefl,fvs,fvl;
#ifdef FLOAT_SQRT
					  double vl;
#endif
					  if (usereverse)
					    {
					      fvx=((-cnx0)>>(TRIANGLE_SHIFT-1-FIX_SHIFT))-
						(lamps[ilamp].x>>(TRIANGLE_SHIFT-FIX_SHIFT));
					      fvy=((-cny0)>>(TRIANGLE_SHIFT-1-FIX_SHIFT))-
						(lamps[ilamp].y>>(TRIANGLE_SHIFT-FIX_SHIFT));
					      fvz=((-cnz0)>>(TRIANGLE_SHIFT-1-FIX_SHIFT))-
						(lamps[ilamp].z>>(TRIANGLE_SHIFT-FIX_SHIFT));
					    }
					  else
					    {
					      fvx=(cnx0>>(TRIANGLE_SHIFT-1-FIX_SHIFT))-
						(lamps[ilamp].x>>(TRIANGLE_SHIFT-FIX_SHIFT));
					      fvy=(cny0>>(TRIANGLE_SHIFT-1-FIX_SHIFT))-
						(lamps[ilamp].y>>(TRIANGLE_SHIFT-FIX_SHIFT));
					      fvz=(cnz0>>(TRIANGLE_SHIFT-1-FIX_SHIFT))-
						(lamps[ilamp].z>>(TRIANGLE_SHIFT-FIX_SHIFT));
					    }
					  fvs=((fvx*fvx)>>FIX_SHIFT)+
					    ((fvy*fvy)>>FIX_SHIFT)+
					    ((fvz*fvz)>>FIX_SHIFT);

					  /* Square root... */
					  fvl=fvs;
					  if (fvl>0)
					    {
#ifndef FLOAT_SQRT
					      ttype fvlold;
					      do {
						fvlold=fvl;
						fvl=(fvlold>>1)+(fvs<<(FIX_SHIFT-1))/fvlold;
					      } while (((fvl-fvlold)>10)||(fvl-fvlold)<-10);
#endif
#ifdef FLOAT_SQRT
					      vl=sqrt((double)(fvs)/FIX_PRECISION);
					      fvl=(int)(vl*FIX_PRECISION);
#endif
					      fvz=(fvz<<FIX_SHIFT)/fvl;

					    }
					  else
					    {
					      fvz=0;
					    }
					  if (usereverse)
					    {
					      skaldiff=(((lamps[ilamp].x>>(TRIANGLE_SHIFT-FIX_SHIFT))*
							 ((-cnx0)>>(TRIANGLE_SHIFT-FIX_SHIFT)))>>FIX_SHIFT)+
						(((lamps[ilamp].y>>(TRIANGLE_SHIFT-FIX_SHIFT))*
						  ((-cny0)>>(TRIANGLE_SHIFT-FIX_SHIFT)))>>FIX_SHIFT)+
						(((lamps[ilamp].z>>(TRIANGLE_SHIFT-FIX_SHIFT))*
						  ((-cnz0)>>(TRIANGLE_SHIFT-FIX_SHIFT)))>>FIX_SHIFT);
					    }
					  else
					    {
					      skaldiff=(((lamps[ilamp].x>>(TRIANGLE_SHIFT-FIX_SHIFT))*
							 (cnx0>>(TRIANGLE_SHIFT-FIX_SHIFT)))>>FIX_SHIFT)+
						(((lamps[ilamp].y>>(TRIANGLE_SHIFT-FIX_SHIFT))*
						  (cny0>>(TRIANGLE_SHIFT-FIX_SHIFT)))>>FIX_SHIFT)+
						(((lamps[ilamp].z>>(TRIANGLE_SHIFT-FIX_SHIFT))*
						  (cnz0>>(TRIANGLE_SHIFT-FIX_SHIFT)))>>FIX_SHIFT);
					    }					  
					  if (skaldiff<0)
					    skaldiff=0;

					  skalrefl=0;
					  if (fvz>=0)
					    {
					      /* skalrefl=pow(vz,myn); */
					      ttype cpow=fvz;
					      ttype myn2=myn;
					      skalrefl=1<<FIX_SHIFT;
					      while (myn2!=0)
						{
						  if (myn2 & 0x1)
						    skalrefl=(skalrefl*cpow)>>FIX_SHIFT;
						  myn2>>=1;
						  cpow=(cpow*cpow)>>FIX_SHIFT;
						}
					      /* skalrefl=fvz; */
					    }

					  col_r+=lamps[ilamp].r*
					    ((((myr<<(FIX_SHIFT-8))*skaldiff)>>
					      (FIX_SHIFT*2-TRIANGLE_SHIFT)));
					  col_g+=lamps[ilamp].g*
					    ((((myg<<(FIX_SHIFT-8))*skaldiff)>>
					      (FIX_SHIFT*2-TRIANGLE_SHIFT)));
					  col_b+=lamps[ilamp].b*
					    ((((myb<<(FIX_SHIFT-8))*skaldiff)>>
					      (FIX_SHIFT*2-TRIANGLE_SHIFT)));

					  /* Specular reflection */

					  col_r_spec+=lamps[ilamp].r*
					    (((cfr0>>(TRIANGLE_SHIFT-FIX_SHIFT))*skalrefl)>>
					      (FIX_SHIFT*2-TRIANGLE_SHIFT));
					  col_g_spec+=lamps[ilamp].g*
					    (((cfr0>>(TRIANGLE_SHIFT-FIX_SHIFT))*skalrefl)>>
					     (FIX_SHIFT*2-TRIANGLE_SHIFT));
					  col_b_spec+=lamps[ilamp].b*
					    (((cfr0>>(TRIANGLE_SHIFT-FIX_SHIFT))*skalrefl)>>
					     (FIX_SHIFT*2-TRIANGLE_SHIFT));
					}
				      col_r>>=TRIANGLE_SHIFT;
				      col_g>>=TRIANGLE_SHIFT;
				      col_b>>=TRIANGLE_SHIFT;
				      oldfbuf=framebuffer[mypoint];


#if 1
				      col_r=((cop0*col_r)+((TRIANGLE_PRECISION-cop0)*(oldfbuf & 0xff)))>>TRIANGLE_SHIFT;
				      col_g=((cop0*col_g)+((TRIANGLE_PRECISION-cop0)*((oldfbuf>>8) & 0xff)))>>TRIANGLE_SHIFT;
				      col_b=((cop0*col_b)+((TRIANGLE_PRECISION-cop0)*((oldfbuf>>16) & 0xff)))>>TRIANGLE_SHIFT;

				      /* Add specular reflection. Not affected by transparency! */
				      col_r=((col_r<<TRIANGLE_SHIFT)+col_r_spec)>>TRIANGLE_SHIFT;
				      col_g=((col_g<<TRIANGLE_SHIFT)+col_g_spec)>>TRIANGLE_SHIFT;
				      col_b=((col_b<<TRIANGLE_SHIFT)+col_b_spec)>>TRIANGLE_SHIFT;
#endif


#if 0
				      col_r=((cop0*col_r)+((TRIANGLE_PRECISION-cop0)*(oldfbuf & 0xff)))>>TRIANGLE_SHIFT;
				      col_g=((cop0*col_g)+((TRIANGLE_PRECISION-cop0)*((oldfbuf>>8) & 0xff)))>>TRIANGLE_SHIFT;
				      col_b=((cop0*col_b)+((TRIANGLE_PRECISION-cop0)*((oldfbuf>>16) & 0xff)))>>TRIANGLE_SHIFT;


				      /* Add specular reflection. Not affected by transparency! */
				      col_r=col_r+(col_r_spec>>TRIANGLE_SHIFT);
				      col_g=col_g+(col_g_spec>>TRIANGLE_SHIFT);
				      col_b=col_b+(col_b_spec>>TRIANGLE_SHIFT);
#endif

				      if (col_r>255)
					col_r=255;
				      if (col_g>255)
					col_g=255;
				      if (col_b>255)
					col_b=255;

				      if (use_fogging)
					{
					  /* Fog factor */
					  int ff=fogtable[myz-ZMINP];
					  col_r=((col_r*ff)+bkgrcol[0]*(FIX_PRECISION-ff))>>FIX_SHIFT;
					  col_g=((col_g*ff)+bkgrcol[1]*(FIX_PRECISION-ff))>>FIX_SHIFT;
					  col_b=((col_b*ff)+bkgrcol[2]*(FIX_PRECISION-ff))>>FIX_SHIFT;
					}
				      if (fade_param!=255)
					{
					  col_r=((col_r*fade_param)+bkgrcol[0]*(255-fade_param))>>8;
					  col_g=((col_g*fade_param)+bkgrcol[1]*(255-fade_param))>>8;
					  col_b=((col_b*fade_param)+bkgrcol[2]*(255-fade_param))>>8;
					}
				      framebuffer[mypoint]=(col_r)|((col_g)<<8)|((col_b)<<16);
				    }
				}
			    }

			  /* interpolate */
			  cz0+=czdiff;
			  /* normals  */
			  cnx0+=cnxdiff;
			  cny0+=cnydiff;
			  cnz0+=cnzdiff;

#ifdef HALF_P
			  if (mypoint&HALF_P_MASK)
			    {
#endif
			      /* exp */
			      cn0+=cndiff;
			      /* rgb */
			      cr0+=crdiff;
			      cg0+=cgdiff;
			      cb0+=cbdiff;
			      /* refl.coeff. */
			      cfr0+=cfrdiff;
			      cop0+=copdiff;
#ifdef HALF_P

			    }
#endif

			  mypoint++;
			}
		    }
		  




		  cline0x+=cline0xdiff;
		  cline1x+=cline1xdiff;

		  /* interpolate */
		  cline0z+=cline0zdiff;
		  cline1z+=cline1zdiff;

#ifdef HALF_P

		  if (yloop&HALF_P_MASK)
		    {

#endif

		      /* exp */
		      cline0n+=cline0ndiff;
		      cline1n+=cline1ndiff;
		      /* rgb */
		      cline0r+=cline0rdiff;
		      cline1r+=cline1rdiff;
		      cline0g+=cline0gdiff;
		      cline1g+=cline1gdiff;
		      cline0b+=cline0bdiff;
		      cline1b+=cline1bdiff;
		      /* refl.coeff. */
		      cline0fr+=cline0frdiff;
		      cline1fr+=cline1frdiff;
		      /* opacity */
		      cline0op+=cline0opdiff;
		      cline1op+=cline1opdiff;

#ifdef HALF_P
	      
		    }

#endif

		  /* normals */
		  cline0nx+=cline0nxdiff;
		  cline1nx+=cline1nxdiff;
		  cline0ny+=cline0nydiff;
		  cline1ny+=cline1nydiff;
		  cline0nz+=cline0nzdiff;
		  cline1nz+=cline1nzdiff;

		  cline0y++;
		  cline1y++;
		}
	    }
	}
    }
}

static double camera_k=0.5;
static double ZMIN=0.1;
static double ZMAX=0.9;


double getck()
{
  return camera_k;
}

double getck_()
{
    return getck();
}


void setck(double ck)
{
  camera_k=ck;
}

void setZMIN(double x)
{
    ZMIN=x;
}

double getZMIN()
{
    return ZMIN;
}

void setZMAX(double x)
{
    ZMAX=x;
}

double getZMAX()
{
    return ZMAX;
}


static void perspectives(short int *x,short int *y,short int *z,int xsize,int ysize)
{
  if (use_perspective)
    {
      int xx=*x,yy=*y,zz=*z;
      int xmid=xsize/2;
      int ymid=ysize/2;
      int ZMAXP=(xsize>ysize) ? xsize : ysize;
      int ZMINP=-ZMAXP;
      double dv=1./(ZMAXP-ZMINP);
      double zv;
      zv=1.-dv*(zz-ZMINP);
      
      xx=xmid+(int)((xx-xmid)*camera_k/zv);
      yy=ymid+(int)((yy-ymid)*camera_k/zv);
      zv=(1-(ZMIN/zv))/(1-ZMIN/ZMAX);
      zz=ZMAXP-(int)(zv*(ZMAXP-ZMINP));
      *x=xx;
      *y=yy;
      *z=zz;
    }
  else
    {
      double ztrans;
      int xmid=xsize/2;
      int ymid=ysize/2;
      gztr_(&ztrans);
      *x=xmid+(int)((*x-xmid)*(1.+(ztrans*0.1)));
      *y=ymid+(int)((*y-ymid)*(1.+(ztrans*0.1)));
    }
}

static void perspectivef(float *x,float *y,float *z,int xsize,int ysize)
{
  if (use_perspective)
    {
      float xx=*x,yy=*y,zz=*z;
      float xmid=xsize/2;
      float ymid=ysize/2;
      float ZMAXP=(xsize>ysize) ? xsize : ysize;
      float ZMINP=-ZMAXP;
      double dv=1./(ZMAXP-ZMINP);
      double zv;
      zv=1.-dv*(zz-ZMINP);
  
      xx=xmid+(xx-xmid)*camera_k/zv;
      yy=ymid+(yy-ymid)*camera_k/zv;
      zv=(1-(ZMIN/zv))/(1-ZMIN/ZMAX);
      zz=ZMAXP-zv*(ZMAXP-ZMINP);
      *x=xx;
      *y=yy;
      *z=zz;
    }
  else
    {
      double ztrans;
      float xmid=xsize/2;
      float ymid=ysize/2;
      gztr_(&ztrans);
      *x=xmid+(*x-xmid)*(1.+(ztrans*0.1));
      *y=ymid+(*y-ymid)*(1.+(ztrans*0.1));
    }
}

static int clip(int z,int xsize,int ysize)
{
  float ZMAXP=(xsize>ysize) ? xsize : ysize;
  float ZMINP=-ZMAXP;
  double dv=1./(ZMAXP-ZMINP);
  double zv;
  zv=1.-dv*(z-ZMINP);
  if (zv>ZMAX) return(1);
  if (zv<ZMIN) return(1);
  return(0);
}

static void normalized(double *x,double *y,double *z)
{
  double l=1./sqrt(*x**x+*y**y+*z**z);
  *x*=l;
  *y*=l;
  *z*=l;
}

static void normalizef(float *x,float *y,float *z)
{
  float l=1./sqrt(*x**x+*y**y+*z**z);
  *x*=l;
  *y*=l;
  *z*=l;
}

/* Draws a fast but slightly approximate sphere (not completely z correct)
   Hmm. It is quite uncorrect with respect to perspective distorsion.
   It is assumed that a sphere is not really affected by perspective.
   This is good for small spheres with not a too wide angle.
   It doesn't really hold here... */
static void draw_sphere(sphere *s,unsigned int *framebuffer,int *zbuffer,
			int xsize,int ysize,int ybankstart,int ybankend)
{
  int use_fogging=get_use_fogging();
  int fade_param=get_fading();
  int bkgrcol[3];
  short int x1,y1,z1,x2,y2,z2,zcorr;
  float fx1=s->x-s->rad;
  float fy1=s->y-s->rad;
  float fz1=s->z;
  float fx2=s->x+s->rad;
  float fy2=s->y+s->rad;
  float fz2=s->z;
  float fdummyx=0;
  float fdummyy=0;
  float fzcorr=s->z+s->rad;
  int xpos,ypos,zpos,rrad,crad;
  get_bkgr_color_vector(&bkgrcol[0],&bkgrcol[1],&bkgrcol[2]);
  perspectivef(&fx1,&fy1,&fz1,xsize,ysize);
  perspectivef(&fx2,&fy2,&fz2,xsize,ysize);
  perspectivef(&fdummyx,&fdummyy,&fzcorr,xsize,ysize);

  x1=(int)fx1+ixtrans;
  y1=(int)fy1;
  z1=(int)fz1;
  x2=(int)fx2+ixtrans;
  y2=(int)fy2;
  z2=(int)fz2;
  zcorr=(int)fzcorr;

  xpos=(x1+x2)>>1;
  ypos=(y1+y2)>>1;
  zpos=z1;

  rrad=(x2-x1)>>1;
  if (rrad==0)
    return;
  crad=((zcorr-zpos)<<FIX_SHIFT)/rrad;

  if ((y2>=ybankstart) && (y1<=ybankend) && (x2>=0) && (x1<xsize))
    {
      int irad2=rrad*rrad;
      int iymin=-rrad;
      int iymax=rrad;
      int iy;
      if ((ypos+iymin)<ybankstart)
	iymin=ybankstart-ypos;
      if ((ypos+iymax)>ybankend)
	iymax=ybankend-ypos;
      
      for (iy=iymin; iy<=iymax; iy++)
	{
	  int ipixelposy=(iy+ypos-ybankstart)*xsize;
	  /* Use integer square root here instead!!! */

#ifdef FLOAT_SQRT
	  int ixmax=(int)(sqrt(irad2-iy*iy));
#endif

	  ttype ixmin;
	  ttype ix;
	  
#ifndef FLOAT_SQRT
	  ttype ds=(irad2-iy*iy)<<FIX_SHIFT2;
	  ttype ixmax=ds;
	  if (ds>0)
	    {
	      ttype dold;
	      ds<<=FIX_SHIFT2-1;
	      do {
		dold=ixmax;
	        ixmax=(ixmax>>1)+ds/ixmax;
	      } while (((ixmax-dold)>=(1<<FIX_SHIFT2))||((ixmax-dold)<=-(1<<FIX_SHIFT2)));
	      ixmax>>=FIX_SHIFT2;
	    }
#endif
	  ixmin=-ixmax;
	  if ((xpos+ixmin)<0)
	    ixmin=-xpos;
	  if ((xpos+ixmax)>=xsize)
	    ixmax=xsize-xpos-1;
	  for (ix=ixmin; ix<=ixmax; ix++)
	    {
	      int ipixelpos;
	      ttype idepth;
#ifdef FLOAT_SQRT
	      ttype depth=(int)(sqrt(irad2-ix*ix-iy*iy));
#endif
#ifndef FLOAT_SQRT
	      

	      ttype ds=(irad2-ix*ix-iy*iy)<<FIX_SHIFT2;
	      ttype depth=ds;
	      if (ds>0)
		{
		  ttype dold;
		  ds<<=FIX_SHIFT2-1;
		  do {
		    dold=depth;
		    depth=(depth>>1)+ds/depth;
		  } while (((depth-dold)>=(1<<FIX_SHIFT2))||((depth-dold)<=-(1<<FIX_SHIFT2)));
		  depth>>=FIX_SHIFT2;
		}
#endif


	      ipixelpos=ix+xpos+ipixelposy;
	      idepth=zpos+((depth*crad)>>FIX_SHIFT);
	      if (idepth>zbuffer[ipixelpos])
		{
		  double nlen=sqrt(ix*ix+iy*iy+depth*depth);
		  ttype cnx0=(int)((ix<<TRIANGLE_SHIFT)/nlen);
		  ttype cny0=(int)((iy<<TRIANGLE_SHIFT)/nlen);
		  ttype cnz0=(int)((depth<<TRIANGLE_SHIFT)/nlen);

		  ttype cfr0=(int)(TRIANGLE_PRECISION*s->fr);

		  ttype myn=s->n;
		  /* rgb */
		  ttype myr=s->r;
		  ttype myg=s->g;
		  ttype myb=s->b;


		  /* compute color */
		  ttype col_r=bkgr_r*cfr0;
		  ttype col_g=bkgr_g*cfr0;
		  ttype col_b=bkgr_b*cfr0;
		  int ilamp;
		  zbuffer[ipixelpos]=idepth;

		 
		  for (ilamp=0; ilamp<nlamps; ilamp++)
		    {
		      ttype fvx=(cnx0>>(TRIANGLE_SHIFT-1-FIX_SHIFT))-
			(lamps[ilamp].x>>(TRIANGLE_SHIFT-FIX_SHIFT));
		      ttype fvy=(cny0>>(TRIANGLE_SHIFT-1-FIX_SHIFT))-
			(lamps[ilamp].y>>(TRIANGLE_SHIFT-FIX_SHIFT));
		      ttype fvz=(cnz0>>(TRIANGLE_SHIFT-1-FIX_SHIFT))-
			(lamps[ilamp].z>>(TRIANGLE_SHIFT-FIX_SHIFT));

		      ttype fvs=((fvx*fvx)>>FIX_SHIFT)+
			((fvy*fvy)>>FIX_SHIFT)+
			((fvz*fvz)>>FIX_SHIFT);

		      ttype skaldiff,skalrefl;
		      /* Square root... */
		      ttype fvl=fvs;
#ifdef FLOAT_SQRT
		      double vl;
#endif
		      if (fvl>0)
			{
#ifndef FLOAT_SQRT
			  /* If FIX_SHIFT==12. fvs mustn't be larger than 255 */
			  ttype fvlold;
			  do {
			    fvlold=fvl;
			    fvl=(fvlold>>1)+(fvs<<(FIX_SHIFT-1))/fvlold;
			  } while (((fvl-fvlold)>10)||((fvl-fvlold)<-10));
#endif
#ifdef FLOAT_SQRT
			  vl=sqrt((double)(fvs)/FIX_PRECISION);
			  fvl=(int)(vl*FIX_PRECISION);
#endif
			  fvz=(fvz<<FIX_SHIFT)/fvl;

			}
		      else
			{
			  fvz=0;
			}
		      skaldiff=(((lamps[ilamp].x>>(TRIANGLE_SHIFT-FIX_SHIFT))*
				     (cnx0>>(TRIANGLE_SHIFT-FIX_SHIFT)))>>FIX_SHIFT)+
			(((lamps[ilamp].y>>(TRIANGLE_SHIFT-FIX_SHIFT))*
			  (cny0>>(TRIANGLE_SHIFT-FIX_SHIFT)))>>FIX_SHIFT)+
			(((lamps[ilamp].z>>(TRIANGLE_SHIFT-FIX_SHIFT))*
			  (cnz0>>(TRIANGLE_SHIFT-FIX_SHIFT)))>>FIX_SHIFT);

		      if (skaldiff<0)
			skaldiff=0;

		      skalrefl=0;
		      if (fvz>=0)
			{
			  ttype cpow=fvz;
			  ttype myn2=myn;
			  /* skalrefl=pow(vz,myn); */
			  skalrefl=1<<FIX_SHIFT;
			  while (myn2!=0)
			    {
			      if (myn2 & 0x1)
				skalrefl=(skalrefl*cpow)>>FIX_SHIFT;
			      myn2>>=1;
			      cpow=(cpow*cpow)>>FIX_SHIFT;
			    }
			  /* skalrefl=fvz; */
			}

		      col_r+=lamps[ilamp].r*
			((((myr<<(FIX_SHIFT-8))*skaldiff)>>
			  (FIX_SHIFT*2-TRIANGLE_SHIFT))
			 +(((cfr0>>(TRIANGLE_SHIFT-FIX_SHIFT))*skalrefl)>>
			   (FIX_SHIFT*2-TRIANGLE_SHIFT)));
		      col_g+=lamps[ilamp].g*
			((((myg<<(FIX_SHIFT-8))*skaldiff)>>
			  (FIX_SHIFT*2-TRIANGLE_SHIFT))
			 +(((cfr0>>(TRIANGLE_SHIFT-FIX_SHIFT))*skalrefl)>>
			   (FIX_SHIFT*2-TRIANGLE_SHIFT)));
		      col_b+=lamps[ilamp].b*
			((((myb<<(FIX_SHIFT-8))*skaldiff)>>
			  (FIX_SHIFT*2-TRIANGLE_SHIFT))
			 +(((cfr0>>(TRIANGLE_SHIFT-FIX_SHIFT))*skalrefl)>>
			   (FIX_SHIFT*2-TRIANGLE_SHIFT)));
		    }
		  col_r>>=TRIANGLE_SHIFT;
		  col_g>>=TRIANGLE_SHIFT;
		  col_b>>=TRIANGLE_SHIFT;
		  if (col_r>255)
		    col_r=255;
		  if (col_g>255)
		    col_g=255;
		  if (col_b>255)
		    col_b=255;
		  if (use_fogging)
		    {
		      /* Fog factor */
		      int ff=fogtable[idepth-ZMINP];
		      col_r=((col_r*ff)+bkgrcol[0]*(FIX_PRECISION-ff))>>FIX_SHIFT;
		      col_g=((col_g*ff)+bkgrcol[1]*(FIX_PRECISION-ff))>>FIX_SHIFT;
		      col_b=((col_b*ff)+bkgrcol[2]*(FIX_PRECISION-ff))>>FIX_SHIFT;
		    }
		  if (fade_param!=255)
		    {
		      col_r=((col_r*fade_param)+bkgrcol[0]*(255-fade_param))>>8;
		      col_g=((col_g*fade_param)+bkgrcol[1]*(255-fade_param))>>8;
		      col_b=((col_b*fade_param)+bkgrcol[2]*(255-fade_param))>>8;
		    }
		  framebuffer[ipixelpos]=(col_r)|((col_g)<<8)|((col_b)<<16);
		}
	    }
	}
    }
}

void draw_triangles(unsigned int *framebuffer,int *zbuffer,int xsize,int ysize,
		    int ybankstart,int ybankend)
{
#if 0
  printf("Triangles,Triangle points,Spheres,Cylinders: %d, %d, %d,%d\n",ntriangles,ntriangle_points,nspheres,ncylinders);
#endif
#ifdef USEOPENGL
  if (get_use_opengl())
    {
      int zbound=(xsize>ysize) ? xsize : ysize;
      GLfloat mymatrix[16]={ 1.0, 0.0, 0.0, 0.0,
			     0.0, 1.0, 0.0, 0.0,
			     0.0, 0.0,-1.0, 0.0,
			     0.0, 0.0, 0.0, 1.0};

      int imylamps=nlamps;
      int glmaxlights=0;
      GLfloat light[4];
      int i;
      triangle *t;
      sphere *s;

      glPushMatrix();
      glViewport(0,0,xsize,ysize);
      glMatrixMode(GL_PROJECTION);
      glLoadIdentity();
      glOrtho(0,xsize-1,0,ysize-1,-zbound,zbound);
      glMatrixMode(GL_MODELVIEW);

      /*  glMultMatrixf(mymatrix); */


      glGetIntegerv(GL_MAX_LIGHTS,&glmaxlights);
      /* printf("GL Max lights=%d\n",glmaxlights); */
      if (imylamps>glmaxlights)
	imylamps=glmaxlights;


      for (i=0; i<imylamps; i++)
	{
	  light[0]=0.;
	  light[1]=0.;
	  light[2]=0.;
	  light[3]=1.;
	  glLightfv(GL_LIGHT0+i,GL_AMBIENT,light);
	  light[0]=lamps_nofp[i].r*DIV255;
	  light[1]=lamps_nofp[i].g*DIV255;
	  light[2]=lamps_nofp[i].b*DIV255;
	  light[3]=1.;
	  glLightfv(GL_LIGHT0+i,GL_DIFFUSE,light);
	  glLightfv(GL_LIGHT0+i,GL_SPECULAR,light);

	  light[0]=lamps_nofp[i].x;
	  light[1]=lamps_nofp[i].y;
	  light[2]=lamps_nofp[i].z;
	  /* printf("Lamp direction: %f,%f,%f\n",light[0],light[1],light[2]); */
	  light[3]=0.;
	  glLightfv(GL_LIGHT0+i,GL_POSITION,light);

	  glEnable(GL_LIGHT0+i);
	}
      if (imylamps<glmaxlights)
	{
	  int i;
	  for (i=imylamps; i<glmaxlights; i++)
	    glDisable(GL_LIGHT0+i);
	}

      
      /* Draw triangles: */

      t=triangle_hook;
      while (t!=NULL)
	{
	  GLfloat mycol[3];
	  GLfloat myshine[]={10.0};
	  mycol[0]=t->point[0]->r*DIV255;
	  mycol[1]=t->point[0]->g*DIV255;
	  mycol[2]=t->point[0]->b*DIV255;
	  glMaterialfv(GL_FRONT,GL_AMBIENT_AND_DIFFUSE,mycol);
	  glMaterialfv(GL_FRONT,GL_SPECULAR,mycol);
	  glMaterialfv(GL_FRONT,GL_SHININESS,myshine);

#if 1
          glBegin(GL_TRIANGLES);
	  glNormal3f(t->point[0]->nx,t->point[0]->ny,t->point[0]->nz);
	  glVertex3f(t->point[0]->x,(ysize-t->point[0]->y),t->point[0]->z);
	  glNormal3f(t->point[1]->nx,t->point[1]->ny,t->point[1]->nz);
	  glVertex3f(t->point[1]->x,(ysize-t->point[1]->y),t->point[1]->z);
	  glNormal3f(t->point[2]->nx,t->point[2]->ny,t->point[2]->nz);
	  glVertex3f(t->point[2]->x,(ysize-t->point[2]->y),t->point[2]->z);
	  glEnd();
#endif
#if 0
          glBegin(GL_TRIANGLES);
	  glNormal3f(t->point[0]->nx,t->point[0]->ny,t->point[0]->nz);
	  glVertex3f(t->point[0]->x,t->point[0]->y,t->point[0]->z);
	  glNormal3f(t->point[1]->nx,t->point[1]->ny,t->point[1]->nz);
	  glVertex3f(t->point[1]->x,t->point[1]->y,t->point[1]->z);
	  glNormal3f(t->point[2]->nx,t->point[2]->ny,t->point[2]->nz);
	  glVertex3f(t->point[2]->x,t->point[2]->y,t->point[2]->z);
	  glEnd();
#endif


	  t=t->next_triangle;
	}
      /* Draw spheres */
      
      s=sphere_hook;
      while (s!=NULL)
	{
	  int rgb[3],myquality;
	  rgb[0]=s->r;
	  rgb[1]=s->g;
	  rgb[2]=s->b;
	  /* Draw the sphere... */
	  myquality=(int)(sqrt(10*s->rad));
	  if (myquality<4)
	    myquality=4;
	  add_triangle_sphere(s->x,s->y,s->z,s->rad,
			      rgb,s->fr,s->n,myquality,myquality,
			      xsize,ysize,
			      1);

	  s=s->next_sphere;
	}

      /* Draw cylinders */



      glPopMatrix();

    }
  else
#endif
    {

      int il;
      triangle *t;
      sphere *s;
      int use_fogging=get_use_fogging();
      double fog_level=get_fogging_prop();
      double fog_level2=get_fogging_prop2();

      /* Convert lamp data to fixed point */
      lamps=malloc(sizeof( lampfp)*(nlamps));
      for (il=0; il<nlamps; il++)
	{
	  lamps[il].r=lamps_nofp[il].r;
	  lamps[il].g=lamps_nofp[il].g;
	  lamps[il].b=lamps_nofp[il].b;
	  lamps[il].x=(int)(lamps_nofp[il].x*TRIANGLE_PRECISION);
	  lamps[il].y=(int)(lamps_nofp[il].y*TRIANGLE_PRECISION);
	  lamps[il].z=(int)(lamps_nofp[il].z*TRIANGLE_PRECISION);
	}

      if (use_fogging)
	compute_fog_table(xsize,ysize,fog_level,fog_level2);


      s=sphere_hook;
      while (s!=NULL)
	{
	  draw_sphere(s,framebuffer,zbuffer,xsize,ysize,ybankstart,ybankend);
	  s=s->next_sphere;
	}

      t=triangle_hook;
      while (t!=NULL)
	{
	  draw_triangle(t,framebuffer,zbuffer,xsize,ysize,ybankstart,ybankend);
	  t=t->next_triangle;
	}
      free( lamps);

    }

}

void add_triangle_sphere(double x,double y,double z,double r,
			 int rgb[3],double rsv,int n,int quality,int quality2,
			 int xsize,int ysize,
			 int draw_using_opengl)
{
  if (r==0)
    return;
#ifdef USEOPENGL
  if ((!draw_using_opengl) && (get_use_opengl()))
    {
      sphere *s=malloc(sizeof( sphere));
      s->x=x;
      s->y=y;
      s->z=z;
      s->rad=r;
      s->r=rgb[0];
      s->g=rgb[1];
      s->b=rgb[2];
      s->fr=rsv;
      s->n=n;
      register_sphere(s);
    }
  else
#endif
    {
#if 0
      int use_fogging=get_use_fogging();
      double fog_level=get_fogging_prop();
#endif
      if (clip(z+r,xsize,ysize))
	return;
      if (clip(z-r,xsize,ysize))
	return;
      if (!use_fast_sphere_emulation)
	{
	  /* Where's the sphere origo??? */

	  float o_x=x;
	  float o_y=y;
	  float o_z=z;
	  double dr=r;
	  double rinv=1./dr;
	  triangle_point ***tp;
	  int ring;

	  perspectivef(&o_x,&o_y,&o_z,xsize,ysize);

	  /* Create the points */
	  tp=malloc(sizeof( triangle_point **)*(quality));

	  for (ring=0; ring<quality; ring++)
	    {
	      int toloop=quality2;
	      double this_a,this_r;
	      int sector;
	      if ((ring==0) || (ring==(quality-1)))
		{
		  tp[ring]=malloc(sizeof( triangle_point *));
		  toloop=1;
		}
	      else
		tp[ring]=malloc(sizeof( triangle_point *)*(quality2));
      
	      this_a=dr*(1.-cos((PI*ring)/(quality-1)));
	      this_r=sqrt(dr*dr-(dr-this_a)*(dr-this_a));

	      for (sector=0; sector<toloop; sector++)
		{
		  tp[ring][sector]=malloc(sizeof( triangle_point));
		  tp[ring][sector]->y=-r+(int)(this_a);
		  if ((ring==0) || (ring==(quality-1)))
		    {
		      tp[ring][sector]->x=0;
		      tp[ring][sector]->z=0;
		    }
		  else
		    {
		      tp[ring][sector]->x=(cos((TWOPI*sector)/quality2)*this_r);
		      tp[ring][sector]->z=(sin((TWOPI*sector)/quality2)*this_r);
		    }
		  tp[ring][sector]->fr=rsv;
		  tp[ring][sector]->n=n;
		  tp[ring][sector]->r=rgb[0];
		  tp[ring][sector]->g=rgb[1];
		  tp[ring][sector]->b=rgb[2];
		  tp[ring][sector]->x+=x;
		  tp[ring][sector]->y+=y;
		  tp[ring][sector]->z+=z;

		  perspectivef(&tp[ring][sector]->x,
			      &tp[ring][sector]->y,
			      &tp[ring][sector]->z,
			      xsize,ysize);
		  tp[ring][sector]->nx=tp[ring][sector]->x-o_x;
		  tp[ring][sector]->ny=tp[ring][sector]->y-o_y;
		  tp[ring][sector]->nz=tp[ring][sector]->z-o_z;
		  normalizef(&tp[ring][sector]->nx,
			    &tp[ring][sector]->ny,
			    &tp[ring][sector]->nz);
		  tp[ring][sector]->opaq=1.0;
		  tp[ring][sector]->twosided=0;
		  register_triangle_point(tp[ring][sector]);

		}
	    }
	  /* Connect all points */
	 
#ifdef USEOPENGL
	  if (get_use_opengl() && draw_using_opengl)
	    {
	      GLfloat mycol[3];
	      mycol[0]=rgb[0]*DIV255;
	      mycol[1]=rgb[1]*DIV255;
	      mycol[2]=rgb[2]*DIV255;
	      
	      glMaterialfv(GL_FRONT,GL_AMBIENT_AND_DIFFUSE,mycol);
	      glBegin(GL_TRIANGLES);

	    }
#endif



	  for (ring=0; ring<(quality-1); ring++)
	    {
	      int sector;
	      for (sector=0; sector<quality2; sector++)
		{
		  int secn=sector+1;
		  if (secn==quality2)
		    secn=0;

		  if (ring==0)
		    {
#ifdef USEOPENGL
		      if (get_use_opengl() && draw_using_opengl)
			{
#if 0
			  float nx,ny,nz,l;
			  nx=tp[0][0]->nx+tp[1][sector]->nx+tp[1][secn]->nx;
			  ny=tp[0][0]->ny+tp[1][sector]->ny+tp[1][secn]->ny;
			  nz=tp[0][0]->nz+tp[1][sector]->nz+tp[1][secn]->nz;
			  l=sqrt(nx*nx+ny*ny*nz*nz);
			  if (l!=0.0)
			    l=1./l;
			  nx*=l;
			  ny*=l;
			  nz*=l;
			  glNormal3f(nx,ny,nz);
#endif
			  glNormal3f(tp[0][0]->nx,tp[0][0]->ny,tp[0][0]->nz);
			  glVertex3f(tp[0][0]->x,(ysize-tp[0][0]->y),tp[0][0]->z);
			  glNormal3f(tp[1][sector]->nx,tp[1][sector]->ny,tp[1][sector]->nz);
			  glVertex3f(tp[1][sector]->x,(ysize-tp[1][sector]->y),tp[1][sector]->z);
			  glNormal3f(tp[1][secn]->nx,tp[1][secn]->ny,tp[1][secn]->nz);
			  glVertex3f(tp[1][secn]->x,(ysize-tp[1][secn]->y),tp[1][secn]->z);
			}
		      else
			{
#endif
			  triangle *t=malloc(sizeof( triangle));
			  t->point[0]=tp[0][0];
			  t->point[1]=tp[1][sector];
			  t->point[2]=tp[1][secn];
			  register_triangle(t);
#ifdef USEOPENGL
			}
#endif
		    }
		  else if (ring==(quality-2))
		    {
#ifdef USEOPENGL
		      if (draw_using_opengl)
			{
#if 0
			  float nx,ny,nz,l;
			  nx=tp[quality-1][0]->nx+tp[quality-2][sector]->nx+tp[quality-2][secn]->nx;
			  ny=tp[quality-1][0]->ny+tp[quality-2][sector]->ny+tp[quality-2][secn]->ny;
			  nz=tp[quality-1][0]->nz+tp[quality-2][sector]->nz+tp[quality-2][secn]->nz;
			  l=sqrt(nx*nx+ny*ny*nz*nz);
			  if (l!=0.0)
			    l=1./l;
			  nx*=l;
			  ny*=l;
			  nz*=l;
			  glNormal3f(nx,ny,nz);
#endif
			  glNormal3f(tp[quality-1][0]->nx,tp[quality-1][0]->ny,tp[quality-1][0]->nz);
			  glVertex3f(tp[quality-1][0]->x,(ysize-tp[quality-1][0]->y),tp[quality-1][0]->z);
			  glNormal3f(tp[quality-2][sector]->nx,tp[quality-2][sector]->ny,tp[quality-2][sector]->nz);
			  glVertex3f(tp[quality-2][sector]->x,(ysize-tp[quality-2][sector]->y),tp[quality-2][sector]->z);
			  glNormal3f(tp[quality-2][secn]->nx,tp[quality-2][secn]->ny,tp[quality-2][secn]->nz);
			  glVertex3f(tp[quality-2][secn]->x,(ysize-tp[quality-2][secn]->y),tp[quality-2][secn]->z);
			}
		      else
			{
#endif
			  triangle *t=malloc(sizeof( triangle));
			  t->point[0]=tp[quality-1][0];
			  t->point[1]=tp[quality-2][sector];
			  t->point[2]=tp[quality-2][secn];
			  register_triangle(t);
#ifdef USEOPENGL
			}
#endif

		    }
		  else
		    {
#ifdef USEOPENGL
		      if (draw_using_opengl)
			{
#if 0
			  float nx,ny,nz,l;
			  nx=tp[ring][sector]->nx+tp[ring+1][sector]->nx+tp[ring+1][secn]->nx;
			  ny=tp[ring][sector]->ny+tp[ring+1][sector]->ny+tp[ring+1][secn]->ny;
			  nz=tp[ring][sector]->nz+tp[ring+1][sector]->nz+tp[ring+1][secn]->nz;
			  l=sqrt(nx*nx+ny*ny*nz*nz);
			  if (l!=0.0)
			    l=1./l;
			  nx*=l;
			  ny*=l;
			  nz*=l;
			  glNormal3f(nx,ny,nz);
#endif
			  glNormal3f(tp[ring][sector]->nx,tp[ring][sector]->ny,tp[ring][sector]->nz);
			  glVertex3f(tp[ring][sector]->x,(ysize-tp[ring][sector]->y),tp[ring][sector]->z);
			  glNormal3f(tp[ring+1][sector]->nx,tp[ring+1][sector]->ny,tp[ring+1][sector]->nz);
			  glVertex3f(tp[ring+1][sector]->x,(ysize-tp[ring+1][sector]->y),tp[ring+1][sector]->z);
			  glNormal3f(tp[ring+1][secn]->nx,tp[ring+1][secn]->ny,tp[ring+1][secn]->nz);
			  glVertex3f(tp[ring+1][secn]->x,(ysize-tp[ring+1][secn]->y),tp[ring+1][secn]->z);
			}
		      else
			{
#endif
			  triangle *t=malloc(sizeof( triangle));
			  t->point[0]=tp[ring][sector];
			  t->point[1]=tp[ring+1][sector];
			  t->point[2]=tp[ring+1][secn];
			  register_triangle(t);
#ifdef USEOPENGL
			}
#endif

#ifdef USEOPENGL
		      if (draw_using_opengl)
			{
#if 0
			  float nx,ny,nz,l;
			  nx=tp[ring][sector]->nx+tp[ring][secn]->nx+tp[ring+1][secn]->nx;
			  ny=tp[ring][sector]->ny+tp[ring][secn]->ny+tp[ring+1][secn]->ny;
			  nz=tp[ring][sector]->nz+tp[ring][secn]->nz+tp[ring+1][secn]->nz;
			  l=sqrt(nx*nx+ny*ny*nz*nz);
			  if (l!=0.0)
			    l=1./l;
			  nx*=l;
			  ny*=l;
			  nz*=l;
			  glNormal3f(nx,ny,nz);
#endif
			  glNormal3f(tp[ring][sector]->nx,tp[ring][sector]->ny,tp[ring][sector]->nz);			  
			  glVertex3f(tp[ring][sector]->x,(ysize-tp[ring][sector]->y),tp[ring][sector]->z);		  
			  glNormal3f(tp[ring][secn]->nx,tp[ring][secn]->ny,tp[ring][secn]->nz);
			  glVertex3f(tp[ring][secn]->x,(ysize-tp[ring][secn]->y),tp[ring][secn]->z);
			  glNormal3f(tp[ring+1][secn]->nx,tp[ring+1][secn]->ny,tp[ring+1][secn]->nz);
			  glVertex3f(tp[ring+1][secn]->x,(ysize-tp[ring+1][secn]->y),tp[ring+1][secn]->z);

			}
		      else
			{
#endif
			  {
			    triangle *t=malloc(sizeof( triangle));
			    t->point[0]=tp[ring][sector];
			    t->point[1]=tp[ring][secn];
			    t->point[2]=tp[ring+1][secn];
			    register_triangle(t);
			  }
#ifdef USEOPENGL
			}
#endif

		    }
		}
	    }

#ifdef USEOPENGL
	  if (draw_using_opengl)
	    {
	      glEnd();
	    }
#endif


	  /* Remove dynamically allocated array but not the triangles themselves */
	  for (ring=0; ring<quality; ring++)
	    {
	      free( tp[ring]);
	    }
	  free( tp);
	}
      else
	{
	  sphere *s=malloc(sizeof( sphere));
	  s->x=x;
	  s->y=y;
	  s->z=z;
	  s->rad=r;
	  s->r=rgb[0];
	  s->g=rgb[1];
	  s->b=rgb[2];
	  s->fr=rsv;
	  s->n=n;
	  register_sphere(s);
	}
    }
}

static void prepare_matrix(double m[4][4])
{
  int i,j;
  for (i=0; i<=3; i++)
    for (j=0; j<=3; j++)
      if (i==j)
	m[i][j]=1;
      else
	m[i][j]=0;
}

static void multiply_matrix(double target[4][4],double source[4][4])
{
  double new_matrix[4][4];
  int i,j;
  for (i=0; i<=3; i++)
    for (j=0; j<=3; j++)
      {
	int k;
	new_matrix[i][j]=0;
	for (k=0; k<=3 ; k++)
	  new_matrix[i][j]+=source[k][j]*target[i][k];
      }
  for (i=0; i<=3; i++)
    for (j=0; j<=3; j++)
      target[i][j]=new_matrix[i][j];
}

/* |v1|==|v2|==1 */
void create_rotation_matrix(double x1,double y1,double z1, /* v1 */
				   double x2,double y2,double z2, /* v2 */
				   double m[4][4])
{
  /* Rotation axis: */
  double ax,ay,az,l,costheta,sintheta,d;
  double r[4][4];
  /* Vector product */
  ax=y1*z2-z1*y2;
  ay=z1*x2-x1*z2;
  az=x1*y2-y1*x2;
  l=sqrt(ax*ax+ay*ay+az*az);
  prepare_matrix(m);
  if (l!=0)
    {
      ax/=l;
      ay/=l;
      az/=l;
    }
  else
    {
      az=1;
    }
  costheta=x1*x2+y1*y2+z1*z2;
  sintheta=sqrt(1-costheta*costheta);
  /* double theta=asin(l);
     printf("Rotation axis: length=%f, costheta=%f, sintheta=%f\n",l,costheta,sintheta); */ 

  d=sqrt(ax*ax+ay*ay);
  /* R1 */
  if (d!=0)
    {
      prepare_matrix(r);      
      r[0][0]=ax/d;
      r[0][1]=-ay/d;
      r[1][0]=ay/d;
      r[1][1]=ax/d;
      multiply_matrix(m,r);
    }
  /* R2 */
  prepare_matrix(r);
  r[0][0]=az;
  r[0][2]=d;
  r[2][0]=-d;
  r[2][2]=az;
  multiply_matrix(m,r);
  /* R3 */
  prepare_matrix(r);
  r[0][0]=costheta;
  r[0][1]=sintheta;
  r[1][0]=-sintheta;
  r[1][1]=costheta;
  multiply_matrix(m,r);
  /* R-2 */
  prepare_matrix(r);
  r[0][0]=az;
  r[0][2]=-d;
  r[2][0]=d;
  r[2][2]=az;
  multiply_matrix(m,r);
  /* R-1 */
  if (d!=0)
    {
      prepare_matrix(r);      
      r[0][0]=ax/d;
      r[0][1]=ay/d;
      r[1][0]=-ay/d;
      r[1][1]=ax/d;
      multiply_matrix(m,r);
    }
}

void multiply_vectord(double v[3],double m[4][4])
{
  double nv[4];
  int i;
  for (i=0; i<=2; i++)
    nv[i]=v[i];
  nv[3]=1;
  for (i=0; i<=2; i++)
    {
      int j;
      v[i]=0;
      for (j=0; j<=3; j++)
	v[i]+=m[j][i]*nv[j];
    }
}

static void multiply_vectorf(float v[3],double m[4][4])
{
  double nv[4];
  int i;
  for (i=0; i<=2; i++)
    nv[i]=v[i];
  nv[3]=1;
  for (i=0; i<=2; i++)
    {
      int j;
      v[i]=0;
      for (j=0; j<=3; j++)
	v[i]+=m[j][i]*nv[j];
    }
}

static void rotate_triangle_point(triangle_point *tp,double m[4][4])
{
  double vec[3];
  /* coordinate */
  vec[0]=(double)(tp->x);
  vec[1]=(double)(tp->y);
  vec[2]=(double)(tp->z);
  multiply_vectord(vec,m);
  tp->x=(int)(vec[0]);
  tp->y=(int)(vec[1]);
  tp->z=(int)(vec[2]);
#if 0
  /* normal */
  vec[0]=tp->nx;
  vec[1]=tp->ny;
  vec[2]=tp->nz;
  multiply_vectord(vec,m);
  tp->nx=vec[0];
  tp->ny=vec[1];
  tp->nz=vec[2];
#endif
}


void add_triangle_cylinder(double x1,double y1,double z1,double r1,
			   double x2,double y2,double z2,double r2,
			   int rgb1[3],double rsv1,int n1,
			   int rgb2[3],double rsv2,int n2,
			   int quality,int xsize,int ysize,
			   int draw_using_opengl)
{
  if ((r1==0) && (r2==0))
    return;
#if 0
#ifdef USEOPENGL
  if (get_use_opengl())
    {
      cylinder *s=malloc(sizeof( cylinder));
      s->x1=x1;
      s->y1=y1;
      s->z1=z1;
      s->rad1=r1;
      s->x2=x2;
      s->y2=y2;
      s->z2=z2;
      s->rad2=r2;
      s->r1=rgb1[0];
      s->g1=rgb1[1];
      s->b1=rgb1[2];
      s->r2=rgb2[0];
      s->g2=rgb2[1];
      s->b2=rgb2[2];
      s->fr1=rsv1;
      s->fr2=rsv2;
      s->n1=n1;
      s->n2=n2;
      s->quality=quality;
      register_cylinder(s);
    }
  else
#endif
#endif
    {
#if 0
      int use_fogging=get_use_fogging();
      double fog_level=get_fogging_prop();
      int temprgb1[3],temprgb2[3];
#endif

      double dx,dy,dz,l;
      double mymatrix[4][4];      
      int clen,sector;
      float p1x,p1y,p1z,p2x,p2y,p2z;
      triangle_point **cyl_point1,**cyl_point2;
      int i;
      
      /* This clipping is not perfect but good enough */
      if (clip(z1+r1,xsize,ysize))
	return;
      if (clip(z1-r1,xsize,ysize))
	return;
      if (clip(z2+r2,xsize,ysize))
	return;
      if (clip(z2-r2,xsize,ysize))
	return;

      /* Always create the cylinder from 0,0,0 to 0,l,0 where l is the length of the cylinder. */
      dx=x2-x1;
      dy=y2-y1;
      dz=z2-z1;
      l=sqrt(dx*dx+dy*dy+dz*dz);
      if (l==0)
	return;
      dx/=l;
      dy/=l;
      dz/=l;
      /* Create rotation matrix: */


      create_rotation_matrix(1,0,0,dx,dy,dz,mymatrix);

      clen=(int)(l);
      if (clen==0)
	return;


      p1x=(float)(x1);
      p1y=(float)(y1);
      p1z=(float)(z1);
      p2x=(float)(x2);
      p2y=(float)(y2);
      p2z=(float)(z2);
      perspectivef(&p1x,&p1y,&p1z,xsize,ysize);
      perspectivef(&p2x,&p2y,&p2z,xsize,ysize);
  
  
      /* Create the points */
      /* There are 2*(quality+1) points on a cylinder. */
      cyl_point1=malloc(sizeof( triangle_point *)*(quality+1));
      cyl_point2=malloc(sizeof( triangle_point *)*(quality+1));
      for (sector=0; sector<quality; sector++)
	{
	  float sy1,sz1,sy2,sz2;
	  float c1v[3],c2v[3];
	  cyl_point1[sector]=malloc(sizeof( triangle_point));
	  cyl_point2[sector]=malloc(sizeof( triangle_point));

	  sy1=cos((TWOPI*sector)/quality)*r1;
	  sz1=sin((TWOPI*sector)/quality)*r1;
	  sy2=cos((TWOPI*sector)/quality)*r2;
	  sz2=sin((TWOPI*sector)/quality)*r2;

	  c1v[0]=0;
	  c1v[1]=sy1;
	  c1v[2]=sz1;
	  c2v[0]=l;
	  c2v[1]=sy2;
	  c2v[2]=sz2;

	  multiply_vectorf(c1v,mymatrix);
	  multiply_vectorf(c2v,mymatrix);

	  c1v[0]+=(float)(x1);
	  c1v[1]+=(float)(y1);
	  c1v[2]+=(float)(z1);
	  c2v[0]+=(float)(x1);
	  c2v[1]+=(float)(y1);
	  c2v[2]+=(float)(z1);

	  
	  perspectivef(&c1v[0],&c1v[1],&c1v[2],xsize,ysize);
	  perspectivef(&c2v[0],&c2v[1],&c2v[2],xsize,ysize);

	  cyl_point1[sector]->x=c1v[0];
	  cyl_point1[sector]->y=c1v[1];
	  cyl_point1[sector]->z=c1v[2];
	  cyl_point2[sector]->x=c2v[0];
	  cyl_point2[sector]->y=c2v[1];
	  cyl_point2[sector]->z=c2v[2];

	  cyl_point1[sector]->nx=c1v[0]-p1x;
	  cyl_point1[sector]->ny=c1v[1]-p1y;
	  cyl_point1[sector]->nz=c1v[2]-p1z;
	  cyl_point2[sector]->nx=c2v[0]-p2x;
	  cyl_point2[sector]->ny=c2v[1]-p2y;
	  cyl_point2[sector]->nz=c2v[2]-p2z;
	  normalizef(&cyl_point1[sector]->nx,
		    &cyl_point1[sector]->ny,
		    &cyl_point1[sector]->nz);
	  normalizef(&cyl_point2[sector]->nx,
		    &cyl_point2[sector]->ny,
		    &cyl_point2[sector]->nz);

	  cyl_point1[sector]->fr=rsv1;
	  cyl_point2[sector]->fr=rsv2;
	  cyl_point1[sector]->r=rgb1[0];
	  cyl_point1[sector]->g=rgb1[1];
	  cyl_point1[sector]->b=rgb1[2];
	  cyl_point2[sector]->r=rgb2[0];
	  cyl_point2[sector]->g=rgb2[1];
	  cyl_point2[sector]->b=rgb2[2];
	  cyl_point1[sector]->n=n1;
	  cyl_point2[sector]->n=n2;
	  cyl_point1[sector]->opaq=1.0;
	  cyl_point1[sector]->twosided=0;
	  cyl_point2[sector]->opaq=1.0;
	  cyl_point2[sector]->twosided=0;

	  register_triangle_point(cyl_point1[sector]);
	  register_triangle_point(cyl_point2[sector]);
	}
      cyl_point1[quality]=malloc(sizeof( triangle_point));
      cyl_point2[quality]=malloc(sizeof( triangle_point));


      cyl_point1[quality]->x=0;
      cyl_point1[quality]->y=0;
      cyl_point1[quality]->z=0;

      cyl_point2[quality]->x=clen;
      cyl_point2[quality]->y=0;
      cyl_point2[quality]->z=0;
      rotate_triangle_point(cyl_point2[quality],mymatrix);
      cyl_point1[quality]->x+=x1;
      cyl_point1[quality]->y+=y1;
      cyl_point1[quality]->z+=z1;  
      cyl_point2[quality]->x+=x1;
      cyl_point2[quality]->y+=y1;
      cyl_point2[quality]->z+=z1;
  
      perspectivef(&cyl_point1[quality]->x,
		  &cyl_point1[quality]->y,
		  &cyl_point1[quality]->z,
		  xsize,ysize);
      perspectivef(&cyl_point2[quality]->x,
		  &cyl_point2[quality]->y,
		  &cyl_point2[quality]->z,
		  xsize,ysize);
  
      cyl_point1[quality]->nx=cyl_point1[quality]->x-cyl_point2[quality]->x;
      cyl_point1[quality]->ny=cyl_point1[quality]->y-cyl_point2[quality]->y;
      cyl_point1[quality]->nz=cyl_point1[quality]->z-cyl_point2[quality]->z;
      cyl_point2[quality]->nx=cyl_point2[quality]->x-cyl_point1[quality]->x;
      cyl_point2[quality]->ny=cyl_point2[quality]->y-cyl_point1[quality]->y;
      cyl_point2[quality]->nz=cyl_point2[quality]->z-cyl_point1[quality]->z;
      normalizef(&cyl_point1[quality]->nx,
		&cyl_point1[quality]->ny,
		&cyl_point1[quality]->nz);
      normalizef(&cyl_point2[quality]->nx,
		&cyl_point2[quality]->ny,
		&cyl_point2[quality]->nz);

      cyl_point1[quality]->fr=rsv1;
      cyl_point2[quality]->fr=rsv2;

      cyl_point1[quality]->r=rgb1[0];
      cyl_point1[quality]->g=rgb1[1];
      cyl_point1[quality]->b=rgb1[2];
      cyl_point2[quality]->r=rgb2[0];
      cyl_point2[quality]->g=rgb2[1];
      cyl_point2[quality]->b=rgb2[2];
      cyl_point1[quality]->n=n1;
      cyl_point2[quality]->n=n2;


      cyl_point1[quality]->opaq=1.0;
      cyl_point1[quality]->twosided=0;
      cyl_point2[quality]->opaq=1.0;
      cyl_point2[quality]->twosided=0;

      register_triangle_point(cyl_point1[quality]);
      register_triangle_point(cyl_point2[quality]);

      /* Produce triangles of the points */

      for (i=0; i<quality; i++)
	{
	  triangle *t=malloc(sizeof( triangle));
	  t->point[0]=cyl_point1[i];
	  if (i!=quality-1)
	    t->point[1]=cyl_point1[i+1];
	  else
	    t->point[1]=cyl_point1[0];
	  t->point[2]=cyl_point2[i];
	  register_triangle(t);
	  t=malloc(sizeof( triangle));
	  t->point[0]=cyl_point2[i];
	  if (i!=quality-1)
	    {
	      t->point[1]=cyl_point1[i+1];
	      t->point[2]=cyl_point2[i+1];
	    }
	  else
	    {
	      t->point[1]=cyl_point1[0];
	      t->point[2]=cyl_point2[0];
	    }
	  register_triangle(t);

	  t=malloc(sizeof( triangle));
	  t->point[0]=cyl_point1[quality];
	  t->point[1]=cyl_point1[i];
	  if (i!=quality-1)
	    t->point[2]=cyl_point1[i+1];
	  else
	    t->point[2]=cyl_point1[0];
	  register_triangle(t);

	  t=malloc(sizeof( triangle));
	  t->point[0]=cyl_point2[quality];
	  t->point[1]=cyl_point2[i];
	  if (i!=quality-1)
	    t->point[2]=cyl_point2[i+1];
	  else
	    t->point[2]=cyl_point2[0];
	  register_triangle(t);
	}
      free( cyl_point1);
      free(cyl_point2);
    }
}


