/* Open a INET socket at a default port, write the port number to a
   file.  "Poll" the socket using select. If data is available on the
   socket import it. Very restricted import is available for this
   option. */

#if defined(USE_BSDSOCK) || defined(USE_WINSOCK)

#ifndef USE_BSDSOCK
#define USE_BSDSOCK
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <math.h>
#ifdef USE_WINSOCK
#include <winsock.h>
#else /* WINSOCK */
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#define WE_ARE_ON_UNIX
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#define INVALID_SOCKET -1
#define SOCKET int
#define closesocket close
#endif /* WINSOCK */

#ifdef USE_CLOCK_GETTIME
#include <time.h>
#else /* USE_CLOCK_GETTIME */
#ifdef USE_GETTIMEOFDAY
#include <sys/time.h>
#else /* USE_GETTIMEOFDAY */
#ifdef USE_FTIME
#include <sys/timeb.h>
#else
#error Found no available timing mechanism
#endif /* USE_FTIME */
#endif /* USE_GETTIMEOFDAY */
#endif /* USE_CLOCK_GETTIME */

#if (SIZEOF_LONG==4)
typedef unsigned long myuint32_t;
#elif (SIZEOF_INT==4)
typedef unsigned int myuint32_t;
#endif

#include <errno.h>

#ifndef HAS_SOCKLEN_T
typedef int socklen_t;
#endif

#endif /* BSDSOCK or WINSOCK */

#include "Xco.h"
#include "userinterface.h"
#include "ccinterface.h"
#include "worldmemory.h"
#include "sockimport.h"

static int initialized=0;
static int open_connection=0;
static int PORT=0;
static int MBUFSIZE=0;

#define COOKIE_LENGTH 16
static char magic_cookie[COOKIE_LENGTH+1];

#define MYMAXHOSTNAMELEN 256

#ifdef USE_BSDSOCK
typedef struct sockhost_t
{
  unsigned int address;
  struct sockhost_t *next;
} sockhost_t;

static char this_hostname[MYMAXHOSTNAMELEN];
static unsigned int this_address;
static struct hostent *this_host;
static int nhosts,thishostnumber;
static sockhost_t *sockhostlist=NULL;
static unsigned int *sockhosts;
static SOCKET origsock;
static SOCKET serversock;
static FILE *serversockstream=0;

#define MAXRETRY 10

static void read_any_sock(void *msg,int len,SOCKET thesock)
{
    int glen,remain=len;
    char *this_msg=msg;
    int retries=MAXRETRY;
    while (remain>0)
    {
#ifdef USE_WINSOCK
	if ((glen=recv(thesock,this_msg,remain,0))<=0)
#else
	if ((glen=read(thesock,this_msg,remain))<=0)
#endif
	{
	    if (glen<0) /* Error has occured. */
	    {
		if (errno!=EINTR) /* If we got EINTR, we just try again. */
		{
		    perror("SOCK: read");
		    exit(1);
		}
		glen=0;
	    }
	    else
	    {
		fprintf(stderr,"SOCK: EOF reading socket\n");
		retries--;
		if (!retries)
		    exit(1);
	    }
	}
	else
	    retries=MAXRETRY;
#if 0
	printf("%d read %d bytes (socket: %d)\n",SOCK_thishost(),glen,thesock);
#endif
	remain-=glen;
	this_msg+=glen;
    }
}

static void write_any_sock(void *msg,int len,SOCKET thesock)
{
    int glen,remain=len;
    char *this_msg=msg;
    while (remain>0)
    {
#ifdef USE_WINSOCK
	if ((glen=send(thesock,this_msg,remain,0))<=0)
#else
	if ((glen=write(thesock,this_msg,remain))<=0)
#endif
	{
	    if (glen<0) /* Error has occured. */
	    {
		if (errno!=EINTR) /* If we got EINTR, we just try again. */
		{
		    perror("SOCK: write");
		    exit(1);
		}
		glen=0;
	    }
	}
#if 0
	printf("%d wrote %d bytes (socket: %d)\n",SOCK_thishost(),glen,thesock);
#endif
	remain-=glen;
	this_msg+=glen;
    }
}




static void create_server()
{
  struct sockaddr_in serv_addr;
  int on=1;
  int lowat;
  socklen_t length;
  if ((origsock=socket(PF_INET,SOCK_STREAM,0))==INVALID_SOCKET)
    {
      perror("SOCK: create server socket");
      exit(1);
    }

  if (setsockopt(origsock,SOL_SOCKET,SO_REUSEADDR,(char*)&on,sizeof(on))<0)
    {
      perror("SOCK: setsockopt - SO_REUSEADDR");
      closesocket(origsock);
      exit(1);
    }
  if (MBUFSIZE)
    {
      lowat=MBUFSIZE;
      if (setsockopt(origsock,SOL_SOCKET,SO_SNDBUF,(char*)&lowat,sizeof(lowat))<0)
	{
	  perror("SOCK: setsockopt - SO_SNDBUF");
	  closesocket(origsock);
	  exit(1);
	}
      
      lowat=MBUFSIZE;
      if (setsockopt(origsock,SOL_SOCKET,SO_RCVBUF,(char*)&lowat,sizeof(lowat))<0)
	{
	  perror("SOCK: setsockopt - SO_RCVBUF");
	  closesocket(origsock);
	  exit(1);
	}
    }

  serv_addr.sin_family=PF_INET;
  serv_addr.sin_port=htons(PORT); /* Get a free port number from the os */
  serv_addr.sin_addr.s_addr=htonl(INADDR_ANY);
  if ((bind(origsock,(struct sockaddr*) &serv_addr,sizeof(serv_addr)))<0)
    {
      perror("SOCK: bind");
      closesocket(origsock);
      exit(1);
    }
  if (listen(origsock,1)<0)
    {
      perror("SOCK: listen");
      closesocket(origsock);
      exit(1);
    }
  /* Get socket info */
  length=sizeof(serv_addr);
  if (getsockname(origsock,(struct sockaddr*) &serv_addr,&length)<0)
    {
      perror("SOCK: getsockname");
      closesocket(origsock);
      exit(1);
    }
  /* Get the port number from the os */
  PORT=ntohs(serv_addr.sin_port);
#if 0
  printf("I have port: %d\n",PORT);
#endif  
}

/* Returns a random integer beween 0 and n-1 */
static int rand_int(int n)
{
  return (int)floor(rand()*(double)(n-1)/(double)RAND_MAX+0.5);
}

static double get_time()
{
  double mytime;
  /* We first try the POSIX function clock_gettime. If it does not work, we fall back
     on gettimeofday. If this code fails to compile, you may need an extra define to
     get gettimeofday into scope. Unfortunately this differs heavily between OS:s. */
#ifdef USE_CLOCK_GETTIME
  struct timespec tv;
  clock_gettime(CLOCK_REALTIME,&tv);
  mytime=tv.tv_sec+tv.tv_nsec*1e-9;
#else /* USE_CLOCK_GETTIME */
#ifdef USE_GETTIMEOFDAY
  struct timeval tv;
  gettimeofday(&tv,NULL);
  mytime=tv.tv_sec+tv.tv_usec*1e-6;
#else /* USE_GETTIMEOFDAY */
#ifdef USE_FTIME
  struct timeb tp;
  ftime(&tp);
  mytime=tp.time+tp.millitm*1e-3;
#endif /* USE_FTIME */
#endif /* USE_GETTIMEOFDAY */
#endif /* USE_CLOCK_GETTIME */
  return mytime;
}

static void create_magic_cookie()
{
  int i;
  magic_cookie[16]=0;
  srand((unsigned int)(get_time()*100));
  for (i=0; i<16; i++)
    magic_cookie[i]=32+rand_int(127-32);
}

static void close_connection()
{
  printf("Closing connection!\n");
  if (open_connection)
    {
      fclose(serversockstream);
      close(serversock);
      open_connection=0;
      cldta_();
    }
}

static void import_data()
{
  int any=1;
  int iframe;
  int notheratoms;
  int nframes=1,nworld;
  double bound_xmin=1e12,bound_ymin=1e12,bound_zmin=1e12;
  double bound_xmax=-1e12,bound_ymax=-1e12,bound_zmax=-1e12;
  double bound;
  double av_x=0., av_y=0., av_z=0.;
  double mass_x,mass_y,mass_z;
  int no;
  int update_bond_rules=1;
#if 0
  printf("Ready to import data\n");
#endif
#if 0
  if (fscanf(serversockstream,"%d",&nframes)!=1)
    {
      close_connection();
      return;
    }
#endif
#if 0
  printf("nframes=%d\n",nframes);
#endif
  /* The following is a more-or-less direct translation of the fortran
     code, without all options and checks. */
#if defined(USCORE2)
  initialize_memory__(&nframes);
#else
  initialize_memory_(&nframes);
#endif
  nworld=nframes+1;
  notheratoms=0;
  for (iframe=1; iframe<=nframes; iframe++)
    {
      int nbuffer=(iframe%2)+1;
      int iatom;
      int ibond;
      int zero=0;
      int one=1;
      int notherbuffer=3-nbuffer;
      int natoms;
      int ntriangles=0;
      int nexpbond=0;
      int nbufferid,notherbufferid;
      double xlocal=0., ylocal=0., zlocal=0.;
      double weight=0.;
      if (fscanf(serversockstream,"%d",&natoms)!=1)
	{
	  close_connection();
	  return;
	}
      if (natoms==-1)
	{
	  if (fscanf(serversockstream,"%d%d",&natoms,&nexpbond)!=2)
	    {
	      close_connection();
	      return;
	    }
	  update_bond_rules=0;
	}
      if (nbuffer==1)
	{
	  nbufferid=0;
	  notherbufferid=nworld;
	}
      else
	{
	  nbufferid=nworld;
	  notherbufferid=0;
	}
#if defined(USCORE2)
      initialize_frame__(&nbufferid,&natoms,&nexpbond,&ntriangles);
      initialize_frame__(&iframe,&natoms,&nexpbond,&ntriangles);
      set_hasframetext__(&iframe,&zero);
#else
      initialize_frame_(&nbufferid,&natoms,&nexpbond,&ntriangles);      
      initialize_frame_(&iframe,&natoms,&nexpbond,&ntriangles);
      set_hasframetext_(&iframe,&zero);
#endif
      for (iatom=1; iatom<=natoms; iatom++)
	{
	  char buf[30];
	  int id,nr;
	  double xc,yc,zc;
	  double w;
	  int iatom_1=iatom-1;
	  if (fscanf(serversockstream,"%d%d%lf%lf%lf",&id,&nr,&xc,&yc,&zc)!=5)
	    {
	      close_connection();
	      return;
	    }
#ifdef USCORE2
	  set_atomxyz__(&nbufferid,&iatom_1,&xc,&yc,&zc);
	  set_atomnr__(&nbufferid,&iatom_1,&nr);
	  set_atomid__(&nbufferid,&iatom_1,&id);
	  set_atomxyz__(&iframe,&iatom_1,&xc,&yc,&zc);
	  set_atomnr__(&iframe,&iatom_1,&nr);
	  set_atomid__(&iframe,&iatom_1,&id);
	  set_defradweight__(&iframe,&iatom_1,&one);
	  set_defcol__(&iframe,&iatom_1,&one);
	  set_atomdrawstyle__(&iframe,&iatom_1,&one);
	  set_deflabel__(&iframe,&iatom_1,&one);
	  set_atomlabel__(&iframe,&iatom_1,"   ",3);
	  set_atomhasmessage__(&iframe,&iatom_1,&zero);
#else
	  set_atomxyz_(&nbufferid,&iatom_1,&xc,&yc,&zc);
	  set_atomnr_(&nbufferid,&iatom_1,&nr);
	  set_atomid_(&nbufferid,&iatom_1,&id);
	  set_atomxyz_(&iframe,&iatom_1,&xc,&yc,&zc);
	  set_atomnr_(&iframe,&iatom_1,&nr);
	  set_atomid_(&iframe,&iatom_1,&id);
	  set_defradweight_(&iframe,&iatom_1,&one);
	  set_defcol_(&iframe,&iatom_1,&one);
	  set_atomdrawstyle_(&iframe,&iatom_1,&one);
	  set_deflabel_(&iframe,&iatom_1,&zero);
	  sprintf(buf,"%d",id);
	  set_atomlabel_(&iframe,&iatom_1,buf,strlen(buf));
	  set_atomhasmessage_(&iframe,&iatom_1,&zero);
#endif	  
	  if (xc<bound_xmin)
	    bound_xmin=xc;
	  if (yc<bound_ymin)
	    bound_ymin=yc;
	  if (zc<bound_zmin)
	    bound_zmin=zc;
	  if (xc>bound_xmax)
	    bound_xmax=xc;
	  if (yc>bound_ymax)
	    bound_ymax=yc;
	  if (zc>bound_zmax)
	    bound_zmax=zc;
	  gatwgh_(&nr,&w);
	  xlocal+=xc*w;
	  ylocal+=yc*w;
	  zlocal+=zc*w;
	  weight+=w;
	}
      notheratoms=natoms;
      av_x+=xlocal/weight;
      av_y+=ylocal/weight;
      av_z+=zlocal/weight;
      for (ibond=1; ibond<=nexpbond; ibond++)
	{
	  int int1,int2;
	  int int1_1,int2_1;
	  int int3,int4;
	  int ir,ig,ib,n,nslice;
	  double r,g,b;
	  double xrsv,xra;
	  int ibond_1=ibond-1;
	  if (fscanf(serversockstream,"%d%d%lf%lf%lf%lf%d%lf",&int1,&int2,&r,&g,&b,&xrsv,&n,&xra)!=8)
	    {
	      close_connection();
	      return;
	    }
	  int1_1=int1-1;
	  int2_1=int2-1;
	  ir=floor(r*255+0.5);
	  ig=floor(g*255+0.5);
	  ib=floor(b*255+0.5);
	  nslice=0;
#ifdef USCORE2
	  set_drawbond__(&iframe,&ibond_1,&one);
	  get_atomid__(&iframe,&int1_1,&int3);
	  get_atomid__(&iframe,&int2_1,&int4);
	  set_bondp__(&iframe,&ibond_1,&int1,&int2,&int3,&int4);
	  set_bond1__(&iframe,&ibond_1,&int1);
	  set_bond2__(&iframe,&ibond_1,&int2);
	  set_defbcol__(&iframe,&ibond_1,&zero);
	  set_bondr1__(&iframe,&ibond_1,&ir);
	  set_bondg1__(&iframe,&ibond_1,&ig);
	  set_bondb1__(&iframe,&ibond_1,&ib);
	  set_bondrsv1__(&iframe,&ibond_1,&xrsv);
	  set_bondn1__(&iframe,&ibond_1,&n);
	  set_bondnslice__(&iframe,&ibond_1,&nslice);
	  set_bondrad1__(&iframe,&ibond_1,&xra);
#else
	  set_drawbond_(&iframe,&ibond_1,&one);
	  get_atomid_(&iframe,&int1_1,&int3);
	  get_atomid_(&iframe,&int2_1,&int4);
	  set_bondp_(&iframe,&ibond_1,&int1,&int2,&int3,&int4);
	  set_bond1_(&iframe,&ibond_1,&int1);
	  set_bond2_(&iframe,&ibond_1,&int2);
	  set_defbcol_(&iframe,&ibond_1,&zero);
	  set_bondr1_(&iframe,&ibond_1,&ir);
	  set_bondg1_(&iframe,&ibond_1,&ig);
	  set_bondb1_(&iframe,&ibond_1,&ib);
	  set_bondrsv1_(&iframe,&ibond_1,&xrsv);
	  set_bondn1_(&iframe,&ibond_1,&n);
	  set_bondnslice_(&iframe,&ibond_1,&nslice);
	  set_bondrad1_(&iframe,&ibond_1,&xra);
#endif	  
	}
    }
  mass_x=av_x/nframes;
  mass_y=av_y/nframes;
  mass_z=av_z/nframes;
  
  bound_xmin-=mass_x;
  bound_ymin-=mass_y;
  bound_zmin-=mass_z;
  bound_xmax-=mass_x;
  bound_ymax-=mass_y;
  bound_zmax-=mass_z;
  bound=fabs(bound_xmin);
  if (fabs(bound_ymin)>bound) bound=fabs(bound_ymin);
  if (fabs(bound_zmin)>bound) bound=fabs(bound_zmin);
  if (fabs(bound_xmax)>bound) bound=fabs(bound_xmax);
  if (fabs(bound_ymax)>bound) bound=fabs(bound_ymax);
  if (fabs(bound_zmax)>bound) bound=fabs(bound_zmax);
  bound+=0.5;
  sbound_(&bound);
#ifdef USCORE2
  def_unit__(&bound);
#else
  def_unit_(&bound);
#endif
  /* These comments were in import.F:
     This is necessary to make routines that check the nworld buffer work,
     if they try to access it between the exit of this routnine, and before
     the start of the next build_world call
  */
#ifdef USCORE2
  get_frame_is_initialized__(&nworld,&no);
#else
  get_frame_is_initialized_(&nworld,&no);
#endif
  if (no==1)
    {
#ifdef USCORE2
      deinitialize_frame__(&nworld);
#else
      deinitialize_frame_(&nworld);
#endif
    }
  /* Set these fortran variables?:
     mass_x,mass_y,mass_z
  */
  smxyz_(&mass_x,&mass_y,&mass_z);

  /* Ensure update. */
  setany_(&any);
  if (update_bond_rules)
    {
      mabrns_(); /* Apply bond rules */
      maprns_(); /* Apply poly rules */
    }
  wupd_(); /* World update */
}

#endif /* USE_BSDSOCK */


/* Externally visible stuff */
void sockimport_init()
{
#ifdef USE_BSDSOCK
  FILE *of;
#ifdef USE_WINSOCK
  WSADATA wsaData;
  if (WSAStartup(MAKEWORD(1,1),&wsaData))
    {
      printf("No suitable winsock implementation.\n");
      exit(1);
    }
  printf("Found WINSOCK %d.%d\n",LOBYTE(wsaData.wVersion),HIBYTE(wsaData.wVersion));
  if ((LOBYTE(wsaData.wVersion)!=1) || (HIBYTE(wsaData.wVersion)!=1))
    {
      printf("No suitable winsock implementation.\n");
      WSACleanup();
      exit(1);
    }   
#endif /* WINSOCK */
  if (gethostname(this_hostname,MYMAXHOSTNAMELEN))
    {
      perror("SOCK: Cannot gethostname.");
      fflush(stdout);
      return;
    }
  if (!(this_host=gethostbyname(this_hostname)))
    {
      perror("SOCK: Cannot gethostbyname.");
      fflush(stdout);
      return;
    }
  if (this_host->h_length!=4)
    exit(1);
  this_address=ntohl(*(myuint32_t*)this_host->h_addr);
  printf("SOCK: I am host %s with address %d.%d.%d.%d\n",this_host->h_name,
	 (this_address>>24)&0xFF,
	 (this_address>>16)&0xFF,
	 (this_address>>8)&0xFF,
	 this_address&0xFF);
  fflush(stdout);
  create_server();
  create_magic_cookie();
  of=fopen("ymol_sockimport","w");
  if (!of)
    {
      fprintf(stderr,"Cannot open ymol_sockimport for output\n");
      exit(1);
    }
  /* If we are on unix (i.e. not included winsock.h) */
#ifdef WE_ARE_ON_UNIX
  chmod("ymol_sockimport",0600);
#endif
  fprintf(of,"%s\n%s\n%d\n",
	  magic_cookie,this_host->h_name,
	  PORT);
  fclose(of);
  initialized=1;
#endif /* USE_BSDSOCK */
}


void sockimport_deinit()
{
#ifdef USE_BSDSOCK
  if (open_connection)
    {
      fclose(serversockstream);
      closesocket(serversock);
    }
  closesocket(origsock);
#ifdef USE_WINSOCK
  WSACleanup();
#endif /* WINSOCK */
#endif /* BSDSOCK */
}

void sockimport_poll()
{
  static int firstcall=1;
#ifdef USE_BSDSOCK
#if 0
  printf("poll...\n");
  fflush(stdout);
#endif
  if (!initialized)
    {
      if (firstcall)
	{
	  fprintf(stderr,"Must initialize sockimport using sockimport_init before doing sockimport_poll\n");
	  fflush(stderr);
	  firstcall=0;
	}
    }
  else
    {
      if (!open_connection)
	{
	  /* Use select to poll the socket for activity */
	  fd_set f;
	  int sv;
	  struct timeval tv;
	  struct sockaddr_in caddr;
	  socklen_t caddrlen=sizeof(caddr);
	  int on=1;
	  FD_ZERO(&f);
	  FD_SET(origsock,&f);
	  tv.tv_sec=0;
	  tv.tv_usec=0;
	  if ((sv=select(origsock+1,&f,NULL,NULL,&tv)))
	    {
	      char magic[COOKIE_LENGTH+2];
	      if ((serversock=accept(origsock,(struct sockaddr*) &caddr,&caddrlen))<0)
		{
		  perror("SOCK: accept");
		  closesocket(origsock);
		  exit(1);
		}
	      if (setsockopt(serversock,SOL_SOCKET,SO_KEEPALIVE,(char*) &on,sizeof(on))<0)
		{
		  perror("SOCK: setsockopt - SO_KEEPALIVE");
		  exit(1);
		}
	      printf("SOCK: %d.%d.%d.%d -> %s\n",
		     (this_address>>24)&0xFF,
		     (this_address>>16)&0xFF,
		     (this_address>>8)&0xFF,
		     this_address&0xFF,
		     inet_ntoa(caddr.sin_addr));
	      fflush(stdout);
	      serversockstream=fdopen(serversock,"r");
	      /* Read the magic cookie */
	      fgets(magic,COOKIE_LENGTH+2,serversockstream);
	      /* Check if the magic cookie is ok */
	      if (!strncmp(magic_cookie,magic,COOKIE_LENGTH))
		{
		  open_connection=1;
		  itmatr_(); /* Must be done for first frame! */
		  import_data();
		  sdefbr_(); /* Must be done for first frame! */
		}
	      else
		{
		  /* Not ok cookie! */
		  printf("SOCK: Unacceptable magic cookie.\n");
		  fclose(serversockstream);
		  closesocket(serversock);
		}
	    }
	}
      else
	{
	  /* Use select to poll the newly established socket for activity */
	  fd_set f;
	  int sv;
	  struct timeval tv;
	  struct sockaddr_in caddr;
	  socklen_t caddrlen=sizeof(caddr);
	  int on=1;
	  FD_ZERO(&f);
	  FD_SET(serversock,&f);
	  tv.tv_sec=0;
	  tv.tv_usec=0;
	  if ((sv=select(serversock+1,&f,NULL,NULL,&tv)))
	    import_data();
	}
    }
#endif /* USE_BSDSOCK */
}
