/* This is povraywrite.c
   A part of the Ymol program
   Copyright (C) 1997-1998 Daniel Spangberg
   */

static char rcsid[]="$Id: povraywrite.c 93 2007-10-17 10:47:10Z daniels $";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "povraywrite.h"
#include "ccinterface.h"
#include "render_params.h"
#include "triangleworld.h"

static FILE *povfile;
static FILE *incfile;
static FILE *incfile2;

static int writeinc=1;
static int writeinc2=1;
static double camera_shift;

void set_writeinc(int p)
{
  writeinc=p;
}

void set_writeinc2(int p)
{
  writeinc2=p;
}

static float xmin,xmax,ymin,ymax,zmin,zmax;
static int first_time;
static int this_pass;

#define SPHERE 0
#define CYLINDER 1
#define TEXT 2
#define TRIANGLE 3

typedef struct characteristic
{
    int type;
    int r,g,b;
    float radius;
    float opacity;
    struct characteristic *next_char;
} characteristic;

characteristic *mychar=(characteristic *) NULL;

static void delete_list()
{
  characteristic *c=mychar;
  while (c!=NULL)
    {
      characteristic *tmp=c;
      c=c->next_char;
      free( tmp);
    }
  mychar=(characteristic *) NULL;
}

static void add_characteristic(characteristic chr)
{
  int already_there=0;
  characteristic *c=mychar;
  while ((!already_there) && (c!=NULL))
    {
	if ((chr.type==c->type) && (chr.r==c->r) && (chr.g==c->g) && (chr.b==c->b) && (chr.radius==c->radius) && 
	    (fabs(chr.opacity-c->opacity)<1e-2))
	    already_there=1;
	c=c->next_char;
    }
  if (!already_there)
    {
      c=malloc(sizeof( characteristic));
      *c=chr;
      c->next_char=mychar;
      mychar=c;
    }
}


static int get_characteristic_number(characteristic chr)
{
  int count=0;
  characteristic *c=mychar;
  int it_is_there=0;
  while ((!it_is_there) && (c!=NULL))
    {
      count++;
      if ((chr.type==c->type) && (chr.r==c->r) && (chr.g==c->g) && (chr.b==c->b) && (chr.radius==c->radius) 
	  && 	    (fabs(chr.opacity-c->opacity)<1e-2))
	  it_is_there=1;
      c=c->next_char;
    }
  return count;
}

static void write_characteristics()
{
  int count=0;
  characteristic *c=mychar;
  while (c!=NULL)
    {
      count++;
      if (writeinc2)
      {
	  if (c->opacity>0.99)
	      fprintf(incfile2,"#declare Ymol_Char_%d = texture {\n pigment {\n  color rgb <%f,%f,%f>\n }\n finish { Ymol_Finish }\n}\n",count,c->r/255.,c->g/255.,c->b/255.);
	  else
	      fprintf(incfile2,"#declare Ymol_Char_%d = texture {\n pigment {\n  color rgbt <%f,%f,%f,%f>\n }\n finish { Ymol_Finish }\n}\n",count,c->r/255.,c->g/255.,c->b/255.,1.0-c->opacity);

      }
      c=c->next_char;
    }
  printf("Found %d characteristics\n",count);
}

void povray_init_scene(char *filenamebase,int pass,double xc,double yc,double zc, double cshift)
{
  char *myfilename=malloc((strlen(filenamebase)+10));
  camera_shift=cshift;
  this_pass=pass;
  first_time=1;

  if (this_pass==FIRST_PASS)
    {
      if (writeinc)
	{
	  sprintf(myfilename,"%sscene.inc",filenamebase);
	  
	  incfile=fopen(myfilename,"w");
	  fprintf(incfile,"#declare Ymol_Finish = finish {\n reflection .35 specular 1\n ambient .3\n}\n");
	}
      if (writeinc2)
	{
	  sprintf(myfilename,"%s.inc",filenamebase);
	  incfile2=fopen(myfilename,"w");
	}
    }
  else
    {
      sprintf(myfilename,"%s.pov",filenamebase);
      povfile=fopen(myfilename,"w");
      fprintf(povfile,"#include \"%sscene.inc\"\n",filenamebase);
      fprintf(povfile,"#include \"%s.inc\"\n",filenamebase);
    }
  free( myfilename);
  xmin=-xc;
  xmax=xc;
  ymin=-yc;
  ymax=yc;
  zmin=-zc;
  zmax=zc;
}

void povray_write_bond(float x1,float y1,float z1,
		float x2,float y2,float z2,
		float radius,int r,int g,int b)
{
  characteristic c;
  x1=-x1;
  x2=-x2;
  c.type=CYLINDER;
  c.r=r;
  c.g=g;
  c.b=b;
  c.radius=radius;
  c.opacity=1.0;
  if (this_pass==FIRST_PASS)
    {
      add_characteristic(c);
    }
  else
    {
      int cno=get_characteristic_number(c);
      fprintf(povfile,"cylinder\n{\n <%f,%f,%f>,\n <%f,%f,%f>,\n %f\n texture { Ymol_Char_%d }\n}\n",x1,y1,z1,x2,y2,z2,radius,cno);
    }
}

void povray_write_atom(float x1,float y1,float z1,
		float radius,int r,int g,int b)
{
  characteristic c;
  x1=-x1;
  c.type=SPHERE;
  c.r=r;
  c.g=g;
  c.b=b;
  c.opacity=1.0;
  c.radius=radius;
  if (this_pass==FIRST_PASS)
    {
      add_characteristic(c);
    }
  else
    {
      int cno=get_characteristic_number(c);
      fprintf(povfile,"sphere\n{\n <%f,%f,%f>,%f\n texture { Ymol_Char_%d }\n}\n",x1,y1,z1,radius,cno);
    }
}

void povray_write_triangle(float x1,float y1,float z1,
			   float x2,float y2,float z2,
			   float x3,float y3,float z3,
			   float nx1,float ny1,float nz1,
			   float nx2,float ny2,float nz2,
			   float nx3,float ny3,float nz3,
			   int r,int g,int b,float opacity)
{
  characteristic c;
  x1=-x1;
  x2=-x2;
  x3=-x3;
  nx1=-nx1;
  nx2=-nx2;
  nx3=-nx3;
  c.type=TRIANGLE;
  c.r=r;
  c.g=g;
  c.b=b;
  c.radius=0.;
  c.opacity=opacity;
  if (this_pass==FIRST_PASS)
    {
      add_characteristic(c);
    }
  else
    {
      int cno=get_characteristic_number(c);
      fprintf(povfile,"smooth_triangle\n{\n <%f,%f,%f>,<%f,%f,%f>\n  <%f,%f,%f>,<%f,%f,%f>\n  <%f,%f,%f>,<%f,%f,%f>\n texture { Ymol_Char_%d }\n}\n",
	      x1,y1,z1,nx1,ny1,nz1,
	      x2,y2,z2,nx2,ny2,nz2,
	      x3,y3,z3,nx3,ny3,nz3,cno);
    }
}

void povray_write_text(float x1,float y1,float z1,float radius,int r,int g,int b,char *text)
{
  characteristic c;
  x1=-x1;
  c.type=TEXT;
  c.r=r;
  c.g=g;
  c.b=b;
  c.opacity=1.0;
  /* radius*=2; */
  c.radius=radius;
  if (this_pass==FIRST_PASS)
    {
      add_characteristic(c);
      c.type=TEXT;
      c.r=0;
      c.g=0;
      c.b=0;
      c.opacity=1.0;
      /* radius*=2; */
      c.radius=radius;
      add_characteristic(c);
    }
  else
    {
      int cno=get_characteristic_number(c);
      fprintf(povfile,"text\n{\n ttf \"timrom.ttf\" \"%s\" %f,0 \n texture { Ymol_Char_%d }\n rotate <0,180,0>\n scale %f\n translate <%f,%f,%f>\n}\n",text,radius*0.1,cno,radius,x1+strlen(text)*radius*0.25,y1-radius*0.25,z1+radius+0.1);
      /* Add black shadow */
      c.type=TEXT;
      c.r=0;
      c.g=0;
      c.b=0;
      c.opacity=1.0;
      /* radius*=2; */
      c.radius=radius;
      cno=get_characteristic_number(c);
      fprintf(povfile,"text\n{\n ttf \"timrom.ttf\" \"%s\" %f,0 \n texture { Ymol_Char_%d }\n rotate <0,180,0>\n scale %f\n translate <%f,%f,%f>\n}\n",text,radius*0.1,cno,radius,x1+strlen(text)*radius*0.25-0.1*radius,y1-radius*0.25-0.1*radius,z1+radius+0.05);
    }
}

void povray_write_light_source(float x1,float y1,float z1,
			       int r,int g,int b)
{
  if (this_pass==FIRST_PASS)
    {
      float s;
      float l=sqrt(x1*x1+y1*y1+z1*z1);
      y1=-y1;
      x1/=l;
      y1/=l;
      z1/=l;
      x1=-x1;
      s=10*(zmax-zmin);
      if (writeinc)
	fprintf(incfile,"light_source\n{\n <%f,%f,%f>\n color rgb <%f,%f,%f>\n}\n",x1*s,y1*s,z1*s,r/255.,g/255.,b/255.);
    }
}

void povray_write_background_light(int r,int g,int b)
{
  if (this_pass==FIRST_PASS)
    {
	int use_fogging=get_use_fogging();
	char *fogme="";
	float s=100*(zmax-zmin);
	if (use_fogging)
	    fogme="hollow\n";
	if (writeinc)
	{
	    fprintf(incfile,"plane\n{\n <0,0,-1>,%f\n pigment\n {\n  color rgb <%f,%f,%f>\n }\n%s}\n",s,1.5*r/255.,1.5*g/255.,1.5*b/255.,fogme);
	    if (use_fogging)
	    {
		double fog_level=get_fogging_prop();
		double fog_level2=get_fogging_prop2();
		fprintf(incfile,"fog\n{\n distance %f\n color rgbt <%f,%f,%f,%f>\n turbulence 0.5\n turb_depth 0.3\n}\n",250./fog_level,r/255.,g/255.,b/255.,fog_level2);
	    }
	}
    }
}

void povray_finish_scene()
{
  if (this_pass==FIRST_PASS)
    {
      double zt;
      double my_zmax=ymax,xp,yp;
      printf("Boundary box: %f,%f,%f to %f,%f,%f\n",xmin,ymin,zmin,xmax,ymax,zmax);
      if (xmax>my_zmax)
	my_zmax=xmax;
      if (xmax>ymax)
	{
	  xp=xmax/ymax;
	  yp=1;
	}
      else
	{
	  xp=1;
	  yp=ymax/xmax;
	}
      gztr_(&zt);
      printf("Camera: %f,%f\n",my_zmax,zt);      
      my_zmax*=1.5;
      my_zmax-=zt;
      if (writeinc)
	{
	    double camera_scaling=0.5/getck();
#if 0
	    fprintf(incfile,"camera\n{ location <%f,0,%f>\n look_at <%f,0,0>\n up <%f,%f,0>\n right <%f,0,0>\n}\n",camera_shift*camera_scaling,my_zmax,camera_shift*camera_scaling,camera_shift*camera_scaling,yp*camera_scaling,(camera_shift+xp)*camera_scaling);
#endif
	    fprintf(incfile,"camera\n{ location <%f,0,%f>\n look_at <%f,0,0>\n up <0,%f,0>\n right <%f,0,0>\n}\n",camera_shift*camera_scaling,my_zmax,camera_shift*camera_scaling,yp*camera_scaling,xp*camera_scaling);
#if 0
      fprintf(incfile,"plane\n{\n <0,1,0>,%f\n pigment\n {\n  color rgb <%f,%f,%f>\n }\n}\n",ymin,1.,1.,1.);
#endif
	}
      write_characteristics();
      if (writeinc)
	fclose(incfile);
      if (writeinc2)
	fclose(incfile2);
    }
  else
    {
      fclose(povfile);
      delete_list();
    }
}


