/* This is periodic_table.c
   A part of the Ymol program
   Copyright (C) 1997-1998 Daniel Spangberg
   */

static char rcsid[]="$Id: periodic_table.c 109 2012-01-23 17:32:00Z daniels $";

#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include "Xco.h"

#include "periodic_table.h"

#define PT_BUTTON_WIDTH 30
#define PT_BUTTON_HEIGHT 22
#define PT_SPACE_BETWEEN 2
#define PT_MARGIN 10
#define PT_TOGGLE_EXTRA 10

#define N_ATOMS 109
static char atom_widget_label[N_ATOMS][4]=
{
  "H","He",
  "Li","Be","B","C","N","O","F","Ne",
  "Na","Mg","Al","Si","P","S","Cl","Ar",
  "K","Ca","Sc","Ti","V","Cr","Mn","Fe","Co","Ni","Cu","Zn","Ga","Ge","As","Se","Br","Kr",
  "Rb","Sr","Y","Zr","Nb","Mo","Tc","Ru","Rh","Pd","Ag","Cd","In","Sn","Sb","Te","I","Xe",
  "Cs","Ba",
  "La","Ce","Pr","Nd","Pm","Sm","Eu","Gd","Tb","Dy","Ho","Er","Tm","Yb","Lu",
  "Hf","Ta","W","Re","Os","Ir","Pt","Au","Hg","Tl","Pb","Bi","Po","At","Rn",
  "Fr","Ra",
  "Ac","Th","Pa","U","Np","Pu","Am","Cm","Bk","Cf","Es","Fm","Md","No","Lr",
  "Rf","Db","Sg","Bh","Hs","Mt"
};




/* We'll define the pt as a 18xn square (0-17,0-(n-1)) */
void get_xy_from_atom_nr(int nr,int *xx, int *yy)
{
  int x=*xx;
  int y=*yy;
  if (nr<3)
    {
      y=0;
      x=17*(nr-1);
    }
  else if (nr<19)
    {
      y=1+((nr-3)/8);
      x=((nr-3)%8);
      if (x>=2)
	x+=10;
    }
  else if (nr<55)
    {
      y=3+((nr-19)/18);
      x=((nr-19)%18);
    }
  else
    {
      /* check for lanthanides and actinides */
      if ((nr>=57) && (nr<=71))
	{
	  /* Lanthanides */
	  y=8;
	  x=2+(nr-57);
	}
      else if ((nr>=89) && (nr<=103))
	{
	  /* Actinides */
	  y=9;
	  x=2+(nr-89);
	}
      else
	{
	  if ((nr>=72) && (nr<=88))
	    nr-=14;
	  if ((nr>=104) && (nr<=118))
	    nr-=28;
	  y=5+((nr-55)/18);
	  x=((nr-55)%18);
	}
    }
  *xx=x;
  *yy=y;
}

static XcoObject atom_widget[N_ATOMS];
static int max_selected_atoms;
static void (*ptcb)(int *atomlist);

static void buttoncallback(XcoObject id,XEvent event)
{
  if (event.type==ButtonPress)
    {
      int myatom=1;
      while (atom_widget[myatom-1]!=id)
	myatom++;
      ptcb(&myatom);
    }
}

static void togglecallback(XcoObject id,XEvent event)
{
  if (event.type==ButtonPress)
    {
      int myatom=1;
      int nsel=0;
      int i,dosel;
      while (atom_widget[myatom-1]!=id)
	myatom++;
      for (i=1; i<=N_ATOMS; i++)
	if (XcoGetToggleSelected(atom_widget[i-1]))
	  nsel++;
      dosel=XcoGetToggleSelected(atom_widget[myatom-1]);
      if ((nsel<max_selected_atoms) || (dosel==1))
	XcoSetToggleSelected(atom_widget[myatom-1],1-dosel);
    }
}

/* This function creates the periodic table form */
static XcoObject create_periodic_table(XcoObject begin,void (*pt_callback)(int *atomlist),
				       int maxsel,int *presel,int npresel)
{
  int mywidth,myheight;
  char atom_widget_name[20];
  XcoObject pt,decorbox;
  int atom_nr;
  max_selected_atoms=maxsel;
  ptcb=pt_callback;
  mywidth=((maxsel!=0)*PT_TOGGLE_EXTRA+PT_BUTTON_WIDTH+PT_SPACE_BETWEEN)*18+20;
  myheight=(PT_BUTTON_HEIGHT+PT_SPACE_BETWEEN)*10+10+20;
  pt=XcoCreateHole(begin,0,0,mywidth,myheight);
  decorbox=XcoCreateBox3D(pt,5,5,mywidth-10,myheight-10,2,1);

  
  for (atom_nr=1; atom_nr<=N_ATOMS; atom_nr++)
    {
      int x,y;
      sprintf(atom_widget_name,"atom%d",atom_nr);
      get_xy_from_atom_nr(atom_nr,&x,&y);

      if (max_selected_atoms==0)
	{
	  atom_widget[atom_nr-1]=XcoCreateCommand(decorbox,
						  x*(PT_BUTTON_WIDTH+
						     PT_SPACE_BETWEEN)+5,
						  y*(PT_BUTTON_HEIGHT+
						     PT_SPACE_BETWEEN)+5,
						  atom_widget_label[atom_nr-1],
						  PT_BUTTON_WIDTH,
						  PT_BUTTON_HEIGHT);
	  
	  if (pt_callback!=NULL)
	    XcoAddCallback(atom_widget[atom_nr-1],buttoncallback);

	}
      else
	{
	  XcoObject lbl;
	  XcoObject decor=XcoCreateBox3D(decorbox,
					 x*(PT_BUTTON_WIDTH+
					    PT_SPACE_BETWEEN+
					    PT_TOGGLE_EXTRA)+5,
					 y*(PT_BUTTON_HEIGHT+
					    PT_SPACE_BETWEEN)+
					 (PT_BUTTON_HEIGHT-PT_TOGGLE_EXTRA)/2+5,
					 PT_BUTTON_WIDTH+PT_TOGGLE_EXTRA,
					 PT_BUTTON_HEIGHT,2,1);

	  atom_widget[atom_nr-1]=XcoCreateToggle(decor,4,
						 2+(PT_BUTTON_HEIGHT-PT_TOGGLE_EXTRA)/4,
						 PT_TOGGLE_EXTRA,PT_TOGGLE_EXTRA,
						 PIXEL(255,0,0),
						 0,0);
	  
	  lbl=XcoCreateLabel(decor,4+PT_TOGGLE_EXTRA,2,
				       atom_widget_label[atom_nr-1],0,0);


	  
	  if (pt_callback!=NULL)
	    XcoAddCallback(atom_widget[atom_nr-1],togglecallback);

	}
    }
  if (maxsel!=0)
    {
      int i;
      for (i=1; i<=npresel; i++)
	XcoSetToggleSelected(atom_widget[presel[i-1]-1],1);
    }
  return pt;
}

static int pt_shell_open=0;
static XcoObject pt_shell;

static void pt_close()
{
  XcoDeleteObject(pt_shell);
  pt_shell_open=0;
}

static void close_callback(XcoObject dummy,XEvent event)
{
  if (event.type==ButtonPress)
    {
      pt_close();
    }
}

static void apply_callback(XcoObject dummy,XEvent event)
{
  if (event.type==ButtonPress)
    {
      int nsel=0;
      int i;
      int *myatomlist;
      for (i=1; i<=N_ATOMS; i++)
	if (XcoGetToggleSelected(atom_widget[i-1]))
	  nsel++;
      myatomlist=malloc(sizeof( int)*(nsel+1));
      myatomlist[0]=nsel;
      nsel=1;
      for (i=1; i<=N_ATOMS; i++)
	if (XcoGetToggleSelected(atom_widget[i-1]))
	  myatomlist[nsel++]=i;

      ptcb(myatomlist);
      free( myatomlist);
    }
}

static void dpt(XcoObject id,XEvent event)
{
  if (XcoDeleteWindow(id,event))
    {
      pt_close();
    }
}

void open_periodic_table(char *name,void (*pt_callback)(int *atomlist),int maxsel,int *presel,int npresel)
{
  if (!pt_shell_open)
    {
      XcoObject hole,close;
      int mywidth=((maxsel!=0)*PT_TOGGLE_EXTRA+PT_BUTTON_WIDTH+PT_SPACE_BETWEEN)*18+20;
      int myheight=(PT_BUTTON_HEIGHT+PT_SPACE_BETWEEN)*10+10+20;
      pt_shell=XcoCreateNamedWindow(0,0,mywidth+10,myheight+50,DEFAULT_BACKGROUND,1,-1,name);
      XcoAddCallback(pt_shell,dpt);
      hole=XcoCreateHole(pt_shell,5,5,mywidth,myheight);
      create_periodic_table(hole,pt_callback,maxsel,presel,npresel);
      close=XcoCreateCommand(pt_shell,mywidth-40,myheight+15,"Close",0,0);
      XcoAddCallback(close,close_callback);
      if (maxsel!=0)
	{
	  XcoObject ok=XcoCreateCommand(pt_shell,20,myheight+15,"Apply",0,0);
	  XcoAddCallback(ok,apply_callback);
	}

      pt_shell_open=1;
    }
}
