#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/* Works best if the box is nearly orthorhombic */
static void compute_ideal_number_of_boxes(int nparticles,
					  double desired_particles_per_cell,
					  double L[3],
					  int box[3])
{
    int i;
    double x,y,z;
    double a,b,c;

    a=L[0];
    b=L[1];
    c=L[2];

    x=pow((double)nparticles*a*a/(desired_particles_per_cell*b*c),1./3.);
    y=x*b/a;
    z=x*c/a;

    box[0]=(int)floor(x+0.5);
    box[1]=(int)floor(y+0.5);
    box[2]=(int)floor(z+0.5);
    for (i=0; i<3; i++)
	if (box[i]<1)
	    box[i]=1;

#if 0
    printf("BOX: %d %d %d\n",box[0],box[1],box[2]);
#endif
}

static void cell_init(int natoms,
		      double *L,
		      int box[3],
		      int **cell_list,
		      int **cell_head)
{
    compute_ideal_number_of_boxes(natoms,1.,L,box);
    *cell_list=malloc(natoms*sizeof **cell_list);
    *cell_head=malloc(box[0]*box[1]*box[2]*sizeof **cell_head);
}


static void cell_deinit(int *cell_list,
			int *cell_head)
{
    free(cell_list);
    free(cell_head);
}

static void cell_sort(int natoms,
		      double *L,
		      double *posx,
		      double *posy,
		      double *posz,
		      int cells[3],
		      int *cell_list,
		      int *cell_head)
{
    int i;
    int ncells=cells[0]*cells[1]*cells[2];
    
    double iL[3];
    iL[0]=1./L[0];
    iL[1]=1./L[1];
    iL[2]=1./L[2];
    
    /* Initialize head */
    for (i=0; i<ncells; i++)
	cell_head[i]=-1;

    for (i=0; i<natoms; i++)
    {
	double x=posx[i];
	double y=posy[i];
	double z=posz[i];

	int ix,iy,iz;
	int icell;

	/* Convert to fractional coordinates */

	x*=iL[0];
	y*=iL[1];
	z*=iL[2];

	/* Which box? */
	ix=(int)floor(x*cells[0]);
	iy=(int)floor(y*cells[1]);
	iz=(int)floor(z*cells[2]);
	
	/* Make really sure particle is inside the primary box */
	if (ix<0)
	    ix=0;
	if (ix>=cells[0])
	    ix=cells[0]-1;
	if (iy<0)
	    iy=0;
	if (iy>=cells[1])
	    iy=cells[1]-1;
	if (iz<0)
	    iz=0;
	if (iz>=cells[2])
	    iz=cells[2]-1;

	/* Add to the list */
	icell=(iz*cells[1]+iy)*cells[0]+ix;
	cell_list[i]=cell_head[icell];
	cell_head[icell]=i;
#if 0
	printf("Particle %d is in cell %d\n",i,icell);
#endif
    }
}

static void get_cell_cutoffs(double rmax,
			     double *L,
			     int cells[3],
			     int cellcutoff[3])
{
    int i;
#if 0
    for (i=0; i<3; i++)
	printf("CELLS [%d] = %d  L=%e\n",i,cells[i],L[i]);
#endif

    /* vector distances. */
    for (i=0; i<3; i++)
	cellcutoff[i]=(int)ceil(cells[i]*rmax/L[i])+1;
}

#define MYABS(i) ((i)<0 ? -(i) : (i))

static void get_cell_neighbours(double rmax,
				double *L,
				int cells[3],
				int cellcutoff[3],
				int *cell_neighbours,
				int *pncells)
{
    int icx,icy,icz;
    int jx,jy,jz;
    double vec[3];
    double rmax2=rmax*rmax;
    double icorners[8][3];
    int ncells;

    /* My central corners. */
    for (icz=0; icz<=1; icz++)
	for (icy=0; icy<=1; icy++)
	    for (icx=0; icx<=1; icx++)
	    {
		vec[0]=(double)icx/cells[0];
		vec[1]=(double)icy/cells[1];
		vec[2]=(double)icz/cells[2];
		/* Shape of box. */
		vec[0]*=L[0];
		vec[1]*=L[1];
		vec[2]*=L[2];
		icorners[icz*4+icy*2+icx][0]=vec[0];
		icorners[icz*4+icy*2+icx][1]=vec[1];
		icorners[icz*4+icy*2+icx][2]=vec[2];
	    }

    ncells=0;
    /* Look in neighbouring cells. */
    for (jz=-cellcutoff[2]; jz<=cellcutoff[2]; jz++)
	for (jy=-cellcutoff[1]; jy<=cellcutoff[1]; jy++)
	    for (jx=-cellcutoff[0]; jx<=cellcutoff[0]; jx++)
	    {
		int jcx,jcy,jcz;
		double jcorners[3];
		int addit=0;
		if ((MYABS(jx)<=1) &&
		    (MYABS(jy)<=1) &&
		    (MYABS(jz)<=1))
		    addit=1;
		if (!addit)
		{
		    /* The corners of this box. */
		    for (jcz=0; jcz<=1; jcz++)
			for (jcy=0; jcy<=1; jcy++)
			    for (jcx=0; jcx<=1; jcx++)
			    {
				int j;
				jcorners[0]=(double)(jcx+jx)/cells[0];
				jcorners[1]=(double)(jcy+jy)/cells[1];
				jcorners[2]=(double)(jcz+jz)/cells[2];
				/* Shape of box. */
				jcorners[0]*=L[0];
				jcorners[1]*=L[1];
				jcorners[2]*=L[2];
				for (j=0; j<8; j++)
				{
				    double dx=jcorners[0]-icorners[j][0];
				    double dy=jcorners[1]-icorners[j][1];
				    double dz=jcorners[2]-icorners[j][2];
				    double d2=dx*dx+dy*dy+dz*dz;
				    if (d2<rmax2)
					addit=1;
				}
			    }
		}
		if (addit)
		{
		    /* Put it in the list. */
		    cell_neighbours[ncells*3]=jx;
		    cell_neighbours[ncells*3+1]=jy;
		    cell_neighbours[ncells*3+2]=jz;
		    ncells++;
		}
	    }

    *pncells=ncells;
}

static int get_displaced_cell(int ix,
			      int iy,
			      int iz,
			      int cells[3])
{
    int rval=-1;
    
    if ((ix>=0) && (ix<cells[0]) &&
	(iy>=0) && (iy<cells[1]) &&
	(iz>=0) && (iz<cells[2]))
	rval=(iz*cells[1]+iy)*cells[0]+ix;
    
    return rval;
}

static int *thepairs=NULL;
static int thepairs_alloc=0;

void compute_pairs(double *posx, double *posy, double *posz, int nsites, double rmax, int *npairs)
{
    int pairs=0;
#if 0
    printf("RMAX: %e\n",rmax);
#endif
    if (nsites)
    {
	double pmin[3],pmax[3];
	double rmax2=rmax*rmax;
	int cells[3];
	double L[3];
	double iL[3];
	int i    ;
	int isite;
	int *cell_list=NULL,*cell_head=NULL;
	int *cell_neighbours=NULL;
	int ncell_neighbours;
	int cellcutoff[3];
	if (thepairs_alloc==0)
	{
	    thepairs_alloc=100;
	    thepairs=malloc(thepairs_alloc*2*sizeof *thepairs);
	}
	pmin[0]=posx[0];
	pmin[1]=posy[0];
	pmin[2]=posz[0];
	pmax[0]=posx[0];
	pmax[1]=posy[0];
	pmax[2]=posz[0];
	for (i=0; i<nsites; i++)
	{
	    if (posx[i]<pmin[0])
		pmin[0]=posx[i];
	    if (posx[i]>pmax[0])
		pmax[0]=posx[i];
	    if (posy[i]<pmin[1])
		pmin[1]=posy[i];
	    if (posy[i]>pmax[1])
		pmax[1]=posy[i];
	    if (posz[i]<pmin[2])
		pmin[2]=posz[i];
	    if (posz[i]>pmax[2])
		pmax[2]=posz[i];
	}
	L[0]=pmax[0]-pmin[0];
	L[1]=pmax[1]-pmin[1];
	L[2]=pmax[2]-pmin[2];
	if (L[0]<1)
	    L[0]=1.;
	if (L[1]<1)
	    L[1]=1.;
	if (L[2]<1)
	    L[2]=1.;
	iL[0]=1./L[0];
	iL[1]=1./L[1];
	iL[2]=1./L[2];
	for (i=0; i<nsites; i++)
	{
	    posx[i]-=pmin[0];
	    posy[i]-=pmin[1];
	    posz[i]-=pmin[2];
	}

	cell_init(nsites,L,cells,&cell_list,&cell_head);
    
	cell_sort(nsites,L,posx,posy,posz,cells,cell_list,cell_head);

	get_cell_cutoffs(rmax,L,cells,cellcutoff);

	cell_neighbours=malloc(3*(cellcutoff[0]*2+1)*
			       (cellcutoff[1]*2+1)*
			       (cellcutoff[2]*2+1)*
			       sizeof *cell_neighbours);

	get_cell_neighbours(rmax,L,cells,cellcutoff,
			    cell_neighbours,&ncell_neighbours);

	for (isite=0; isite<nsites; isite++)
	{
	    int ix1,iy1,iz1;
	    double x1=posx[isite];
	    double y1=posy[isite];
	    double z1=posz[isite];
	    int icell_neighbours;
	    /* Which cell? */
	    ix1=(int)floor(x1*iL[0]*cells[0]);
	    iy1=(int)floor(y1*iL[1]*cells[1]);
	    iz1=(int)floor(z1*iL[2]*cells[2]);


	    /* Make sure it is inside */
	    if (ix1<0)
		ix1=0;
	    if (ix1>=cells[0])
		ix1=cells[0]-1;
	    if (iy1<0)
		iy1=0;
	    if (iy1>=cells[1])
		iy1=cells[1]-1;
	    if (iz1<0)
		iz1=0;
	    if (iz1>=cells[2])
		iz1=cells[2]-1;

#if 0
	    printf("processing isite=%d\n",isite);
	    printf("cell=%d %d %d, mycell=%d\n",ix1,iy1,iz1,get_displaced_cell(ix1,iy1,iz1,cells));
	    printf("ncell_neighbours: %d\n",ncell_neighbours);
#endif	
    
	    for (icell_neighbours=0; icell_neighbours<ncell_neighbours; icell_neighbours++)
	    {
		int icell2=get_displaced_cell(ix1+cell_neighbours[icell_neighbours*3],
					      iy1+cell_neighbours[icell_neighbours*3+1],
					      iz1+cell_neighbours[icell_neighbours*3+2],
					      cells);
		if (icell2!=-1)
		{
		    int isite2;
#if 0
		    printf("Particles in delta cell: %d %d %d (icell2=%d)\n",
			   cell_neighbours[icell_neighbours*3],
			   cell_neighbours[icell_neighbours*3+1],
			   cell_neighbours[icell_neighbours*3+2],icell2);
#endif
		    /* List of all particles in this cell */
		    for (isite2=cell_head[icell2]; isite2!=-1; isite2=cell_list[isite2])
		    {
#if 0
			printf("Checking %d %d\n",isite,isite2);
#endif
			if (isite<isite2) /* Only once per pair! */
			{
			    double x2=posx[isite2];
			    double y2=posy[isite2];
			    double z2=posz[isite2];
			    double dx=x2-x1;
			    double dy=y2-y1;
			    double dz=z2-z1;
			    double d2=dx*dx+dy*dy+dz*dz;
#if 0
			    printf("d2: %e, rmax2: %e\n",d2,rmax2);
#endif
			    if (d2<rmax2)
			    {
				if (pairs+1==thepairs_alloc)
				{
				    thepairs_alloc*=1.5;
				    thepairs=realloc(thepairs,thepairs_alloc*2*sizeof *thepairs);
				}
				thepairs[pairs*2]=isite;
				thepairs[pairs*2+1]=isite2;
#if 0
				printf("Adding %d %d\n",isite,isite2);
#endif
				pairs++;
			    }
		 
			}
		    }
		}
	    }
	}
	free(cell_neighbours);
	cell_deinit(cell_list,cell_head);
	for (i=0; i<nsites; i++)
	{
	    posx[i]+=pmin[0];
	    posy[i]+=pmin[1];
	    posz[i]+=pmin[2];
	}
    }
    *npairs=pairs;
}

static void get_pair(int pair, int *pair1, int *pair2)
{
    *pair1=thepairs[pair*2];
    *pair2=thepairs[pair*2+1];
}

void pairgt(int *pair, int *pair1, int *pair2)
{
    get_pair(*pair,pair1,pair2);
}

void pairgt_(int *pair, int *pair1, int *pair2)
{
    get_pair(*pair,pair1,pair2);
}

void pairgt__(int *pair, int *pair1, int *pair2)
{
    get_pair(*pair,pair1,pair2);
}
